
project.field<-function(field,lons,lats,wm,proj="robin",center=0,lonminmax=NULL,latminmax=NULL,lonsmapgrid=seq(0,360,by=60),latsmapgrid=seq(-90,90,by=30),lat_0=0,lat_1=10,lat_2=20,lat_ts=0,more.options=""){
  # projects a 2D or 3D array to a map and creates the same projection of the world-coastline for plotting and the grid-lines
  # RN_2015_12_12
  #  required packages: ggplot2 (detach function), rgdal (project), fields(make.surface.grid)
  # Attention plot fields in the lon order "xo" otherwise the plots are not correct! (i.e. if output is x use field[x$xo,] to plot a 2d field (see below))
  # Input variables:
  #   field:    global grid as 2D or 3D array,  
  #                dim 1 = number of longitudes (nlon)
  #                dim 2 = number of latitudes (nlat)
  #               (dim 3 = years, levels, etc)
  #   lons:     longitudes of the grid (length=nlon)
  #   lats:     latitudes of the grid (length=nlon)
  #   wm:       world map to be plotted over the grid. provided as SpatialLinesDataFrame (S4 object)
  #             a simple world map can be obtained from http://www.naturalearthdata.com/http//www.naturalearthdata.com/download/110m/physical/ne_110m_coastline.zip
  #             the data can then be read in with cl110<-readOGR("C:/Users/neukom/Documents/R/ne_110m_coastline", layer="ne_110m_coastline") requires rgdal-package or cl2<-readShapeLines("C:/Users/neukom/Documents/R/ne_110m_coastline/ne_110m_coastline.shp") requires maptools package and reads in also coordinate reference information
  #   proj:     a projection as text string. common ones are "longlat" or "robin".  list of projections can be get by proj.list<-projInfo("proj") more details with ?mapPlot (oce package)
  #             see also the list at: http://www.remotesensing.org/geotiff/proj_list/
  #             projection examples at https://en.wikipedia.org/wiki/List_of_map_projections
  #              R-examples with all projections in proj-projections_field_0-centered.pdf
  #   center:   longitude at which the map is centered in degrees. can be in the (-180°) to (+180°) or 0° to 360° convention
  #   lonminmax: range of longitudes to be retained for the coastline. e.g. c(-50,50), to avoid coastlines in the white area of the plot
  #              can be in the range -180 to 180 OR 0 to 365.
  #              Note that this only affects the coastline. to limit the coordinates of the plotted field define this in the plot(see example below)
  #   latminmax: same for latitudes
  #    lonsmapgrid: longitudes of the grid-lines
  #    latsmapgrid: latitudes of the grid-lines
  #   lat_0=0,lat_1=10,lat_2=20,lat_ts=0: some projections require these parameters. see more info at ?mapPlot and https://github.com/OSGeo/proj.4/wiki/GenParms
  #   more.option:  string of further options passed to the projection string such as +ellps=.see examples in various CRS functions
  # 
  # Output: list(x) with the following fields
  #   lons: projected longitudes in a 2D array (nlon*nlat)
  #   lats: projected latitudes in a 2D array (nlon*nlat)
  #   field:corresponding 2D or 3D field
  #   wm:   projected worldmap as 2D array with the columns containing the lons and lats to be connected
  #   gridlines: the  projected grid that can be plotted over the map
  #     lonaxat: the positions of the x-xis labels and ticks
  #     lataxat: the positions of the y-xis labels and ticks
  #   xo: often, the order of the longitudes in the plotted field needs to be chagned to be correctly plotted. xo gives you this order. In this case plot field[xo,]
  #
  #   the output can be plottes with image.plot (n-th year in a 3D field), requires fields package:
  #     image.plot2(x$lons,x$lats,x$field[x$xo,,n],col.lab="white",col.lab="white",col.axis="white",tck=0)
  #     lines(x$wm)
  #     lines(x$gridlines)
  #     axis(1,at=x$lonaxat,labels=lonsmapgrid)
  #     axis(2,at=x$lataxat,labels=latsmapgrid)
  #   Note that for prjected grids, the function image.plot2 needs to be used (code below) otherwise a custom color range is not displayed correctly!
  #
  # for polar azimutal projections use "laea" projection and define the latitude limits in the plots. example:
  #   x<-project.field(field,lons,lats,cl2,"laea",center=0,lat_0=-90,latminmax=c(-90,0),more.options="",lonsmapgrid=lonsmapgrid,latsmapgrid=latsmapgrid)
  #   image.plot2(x$lons[,1:18],x$lats[,1:18],x$field[,1:18,1],breaks=breaks,col=col,col.lab="white",col.axis="white",tck=0)
  #   lines(x$wm)
  ############################
  
  xgrid<- make.surface.grid(list(x= lons, y=lats))
  lons.2d<-xgrid[,1]
  lats.2d<-xgrid[,2]
  nlon<-dim(field)[1]
  nlat<-dim(field)[2]
  lons.grid<-array(lons.2d,dim=c(nlon,nlat))
  
  #adapt longitudes to -180 to 180 scale if necessary
  if(max(lons)>180){
    negs<-which(lons>180)
    lons[negs]<-lons[negs]-360
    negs<-which(lons.grid>180)
    lons.grid[negs]<-lons.grid[negs]-360
  }
  
  #create data frame out of  SpatialLinesDataFrame
  wm.df<-fortify(wm)
  wm.df.fix<-sepjumps.df(wm.df)
  
  #create the array for the grid
  gridlons<-rep(lonsmapgrid,each=length(lats)+1)
  gridlats<-rep(c(lats,NA),length(lonsmapgrid))
  gridlons2<-rep(c(lons,NA),length(latsmapgrid))
  gridlats2<-rep(latsmapgrid,each=length(lons)+1)
  gridlines<-rbind(cbind(gridlons,gridlats),cbind(gridlons2,gridlats2))
  
  #make sure center longitude is in right format
  if(center<0) center<-360+center
  
  ##change grid lon order  if necessary
  ds<-(abs(lons-(center-180)))
  mins<-which(ds==min(ds))
  if(length(mins)>1) mins<-mins[2]
  
  if(mins==1){
    newlons<-mins:nlon
  }else{
    newlons<-c((mins:nlon),(1:(mins-1)))  
  }
  if(length(dim(field))==3){
    field2<-field[newlons,,]
  }else{
    field2<-field[newlons,]
  }
  lons.grid2<-lons.grid[newlons,]
  lons.2d2<-as.vector(lons.grid2)
  
  pluslon<-center
  
  #create proj4 text for projection
  projtext<-paste0("+proj=",proj," +lon_0=",pluslon, " +lat_0=",lat_0," +lat_1=",lat_1," +lat_2=",lat_2," +lat_ts=",lat_ts," ",more.options)
  
  #make worldmap projection
  if(length(latminmax)>0){
    wm.df.fix[which(wm.df.fix[,2]<latminmax[1]),c(1,2)]<-NA
    wm.df.fix[which(wm.df.fix[,2]>latminmax[2]),c(1,2)]<-NA
  }
  if(length(lonminmax)>0){
    if(lonminmax[2]>180){
      wm.df.fix[which(wm.df.fix[,1]<lonminmax[1] & wm.df.fix[,1]>=0),c(1,2)]<-NA 
      wm.df.fix[which(wm.df.fix[,1]>(lonminmax[2]-360) & wm.df.fix[,1]<=0),c(1,2)]<-NA
    }else{
      wm.df.fix[which(wm.df.fix[,1]<lonminmax[1]),c(1,2)]<-NA
      wm.df.fix[which(wm.df.fix[,1]>lonminmax[2]),c(1,2)]<-NA
    }
  }
  wm.proj<-project(as.matrix(wm.df.fix[,1:2]),projtext)
  wm.proj.fixed<-sepjumps.lonlat(wm.proj)
  
  #make projection of the grid lines
  gridlines.proj<-project(gridlines,projtext)
  xl<-length(lonsmapgrid)*(length(lats)+1)
  xl2<-seq(1,xl,by=length(lats)+1)
  lonaxat<-gridlines.proj[xl2,1]
  xl3<-seq(xl+1,dim(gridlines)[1],by=length(lons)+1)
  lataxat<-gridlines.proj[xl3,2]
  
  #make field projection
  res <- project(cbind(lons.2d2, lats.2d),projtext)
  projectedLon<- matrix(res[,1],nrow=nlon,ncol=nlat)            # reform into matrices the same sizes as the image
  projectedLat<- matrix(res[,2],nrow=nlon,ncol=nlat)            # reform into matrices the same sizes as the image
  #image.plot(projectedLon,projectedLat,field2[,,1984],breaks=breaks,col=col)
  #lines(wm.proj.fixed)
  
  xo<-newlons
  
  out<-list(lons=projectedLon,lats=projectedLat,field=field2,wm=wm.proj.fixed,gridlines=gridlines.proj,lonaxat=lonaxat,lataxat=lataxat,lonsmapgrid=lonsmapgrid,latsmapgrid=latsmapgrid,xo=xo,newlons=newlons)
}

# avoid lines connecting the different regions if a map that has been converted to a data.frame with fortify is plotted with plot or lines
sepjumps.df<-function(x){
  jumps<-which(diff(as.numeric(x$id))==1)
  cdf2<-x
  for (i in rev(jumps)){
    cdf2<-rbind(cdf2[1:i,],c(NA,NA),cdf2[(i+1):dim(cdf2)[1],])
  }
  cdf2
}

#avoid the lines crossing the map after the center longitude has been changed in a lonlat array
sepjumps.lonlat<-function(x){
  tresh<-(max(x[,1],na.rm=T)-min(x[,1],na.rm=T))*0.5
  jumps<-which(abs(diff(x[,1]))>tresh)
  cdf2<-x
  for (i in rev(jumps)){
    cdf2<-rbind(cdf2[1:i,],c(NA,NA),cdf2[(i+1):dim(cdf2)[1],])
  }
  cdf2
}


##calculate latitude weighted mean of a field
#field must have 3 dimensions: 1=lon,2=lat,3=time
#lat is the latitudes of the field
latweightmean.field<-function(field,lats,sqrt.cos=FALSE){
  weights<-cos(lats*pi/180)
  if(sqrt.cos==T) weights<-weights^0.5
  if(length(which(is.na(field)))==0){
    weighted<-field
    for(la in seq_along(lats)){
      weighted[,la,]<-field[,la,]*weights[la]
    }
    meanw<-apply(weighted,3,sum)/(sum(weights)*dim(field)[1])
  }else{
    meanw<-vector(length=dim(field)[3])
    for(y in 1:dim(field)[3]){
      weightedsum<-0
      weightssum<-0
      for(la in seq_along(lats)){
        for(lo in 1:dim(field)[1]){
          if(!is.na(field[lo,la,y])){
            weightedsum<-weightedsum+field[lo,la,y]*weights[la]
            weightssum<-weightssum+weights[la]
          }
        }
      }
      meanw[y]<-weightedsum/weightssum
    }
  }
  meanw
}


##apply a function to a timeseries-matrix. the outcome is a time series with the same tsp
##fun should look something like this: function(x) mean(x,na.rm=T)
tsapply<-function(x,dim,fun){
  out<-ts(apply(x,dim,fun),start=start(x)[1])
}



#funktion für distanzberechnung
distanz <- function(lat.a,lon.a,lat.b,lon.b){ 
  rad<-0.01745329
  pi2<-pi/2
  lat.a<-lat.a*rad
  lon.a<-lon.a*rad
  lat.b<-lat.b*rad
  lon.b<-lon.b*rad
  if(lon.a==lon.b & lat.a==lat.b){distan<-0}else{
    distan<-sin(lat.a)*sin(lat.b)+cos(abs(lon.a-lon.b))*cos(lat.a)*cos(lat.b) 
    distan<-(atan(-distan/(-distan*distan+1)^(1/2))+pi2)*6378.388
  }
  return(distan)
}


###read a table containing a time series, first column = years
read.ts<-function(filename,sep=";",header=T){
  ind<-as.matrix(read.table(filename,sep=sep,header=header))
  data<-ts(ind[,-1],start=ind[1,1])
  return(data)
}



#modified image.plot function allowing extra plot commands like points, axes etc.
#this extra commands are provided as strings in the withincomands variable 
# this is helpful for multi-panel figures
image.plot3<-function (..., add = FALSE, breaks = NULL, nlevel = 64, col = NULL, 
                       horizontal = FALSE, legend.shrink = 0.9, legend.width = 1.2, 
                       legend.mar = ifelse(horizontal, 3.1, 5.1), legend.lab = NULL, 
                       legend.line = 2, graphics.reset = FALSE, bigplot = NULL, 
                       smallplot = NULL, legend.only = FALSE, lab.breaks = NULL, 
                       axis.args = NULL, legend.args = NULL, legend.cex = 1, midpoint = FALSE, 
                       border = NA, lwd = 1, verbose = FALSE,withincommands) 
{
  old.par <- par(no.readonly = TRUE)
  if (is.null(col)) {
    col <- tim.colors(nlevel)
  }
  else {
    nlevel <- length(col)
  }
  info <- imagePlotInfo(..., breaks = breaks, nlevel = nlevel)
  breaks <- info$breaks
  if (verbose) {
    print(info)
  }
  if (add) {
    big.plot <- old.par$plt
  }
  if (legend.only) {
    graphics.reset <- TRUE
  }
  if (is.null(legend.mar)) {
    legend.mar <- ifelse(horizontal, 3.1, 5.1)
  }
  temp <- imageplot.setup(add = add, legend.shrink = legend.shrink, 
                          legend.width = legend.width, legend.mar = legend.mar, 
                          horizontal = horizontal, bigplot = bigplot, smallplot = smallplot)
  smallplot <- temp$smallplot
  bigplot <- temp$bigplot
  if (!legend.only) {
    if (!add) {
      par(plt = bigplot)
    }
    if (!info$poly.grid) {
      image(..., breaks = breaks, add = add, col = col)
    }
    else {
      poly.image(..., add = add, breaks = breaks, col = col, midpoint = midpoint, 
                 border = border, lwd.poly = lwd)
    }
    big.par <- par(no.readonly = TRUE)
    for(wic in seq_along(withincommands)){
      eval(parse(text=withincommands[wic]))
    }
    
  }
  if ((smallplot[2] < smallplot[1]) | (smallplot[4] < smallplot[3])) {
    #par(old.par)
    #stop("plot region too small to add legend\n")
  }else{
    ix <- 1:2
    iy <- breaks
    nBreaks <- length(breaks)
    midpoints <- (breaks[1:(nBreaks - 1)] + breaks[2:nBreaks])/2
    iz <- matrix(midpoints, nrow = 1, ncol = length(midpoints))
    if (verbose) {
      print(breaks)
      print(midpoints)
      print(ix)
      print(iy)
      print(iz)
      print(col)
    }
    par(new = TRUE, pty = "m", plt = smallplot, err = -1)
    if (!horizontal) {
      image(ix, iy, iz, xaxt = "n", yaxt = "n", xlab = "", 
            ylab = "", col = col, breaks = breaks)
    }
    else {
      image(iy, ix, t(iz), xaxt = "n", yaxt = "n", xlab = "", 
            ylab = "", col = col, breaks = breaks)
    }
    if (!is.null(lab.breaks)) {
      axis.args <- c(list(side = ifelse(horizontal, 1, 4), 
                          mgp = c(3, 1, 0), las = ifelse(horizontal, 0, 2), 
                          at = breaks, labels = lab.breaks), axis.args)
    }
    else {
      axis.args <- c(list(side = ifelse(horizontal, 1, 4), 
                          mgp = c(3, 1, 0), las = ifelse(horizontal, 0, 2)), 
                     axis.args)
    }
    do.call("axis", axis.args)
    box()
    if (!is.null(legend.lab)) {
      legend.args <- list(text = legend.lab, side = ifelse(horizontal, 
                                                           1, 4), line = legend.line, cex = legend.cex)
    }
    if (!is.null(legend.args)) {
      do.call(mtext, legend.args)
    }
    mfg.save <- par()$mfg
    if (graphics.reset | add) {
      par(old.par)
      par(mfg = mfg.save, new = FALSE)
      invisible()
    }
    else {
      par(big.par)
      par(plt = big.par$plt, xpd = FALSE)
      par(mfg = mfg.save, new = FALSE)
      invisible()
    }
  }
}


#add polygon to a timeseries plot
#lower and upper are the timeseries for the lower and upper bounds resp.
tspolygon<-function(lower,upper,col=rgb(0,0,0,0.4),border=NA){
  if(is.na(min(lower))){
    lower<-na.omit(lower)
  }
  if(is.na(min(upper))){
    upper<-na.omit(upper)
  }
  if(is.na(border)) border<-col
  xx<-c(tsp(lower)[1]:tsp(lower)[2],tsp(lower)[2]:tsp(lower)[1])
  polygon(xx,c(lower,rev(upper)),border=border,col=col)
}



###filter time series with custom filter:
tsfilt<-function(x,width=31,method="loess",cut.end=T){
  #possibilities
  #"loess" --> loessfilt function
  #"spline2" --> splinesmoother2 from Dave
  # "gauss" --> gaussfilter
  #"rm" --> running mean
  #"hamming" --> hamming
  #"bw" --> butterworth
  if(method=="loess"){
    filtered<-loessfilt(x,width,cut.end=cut.end)
  }
  if(method=="spline2"){
    if(length(which(is.na(x)))>0){
      if(is.null(dim(x))==T){
        filtered<-splinesmoother2.nas(x,width,cut.end=cut.end)
      }else{ 
        filtered<-ts(apply(x,2,function(y) splinesmoother2.nas(y,width,cut.end)),start=start(x)[1])
      }
    }else{
      filtered<-splinesmoother2(x,width)
      if(cut.end==T){
        sx<-fy(x)+floor(width/2)-1
        ex<-ly(x)-floor((width-0.5)/2)+1
        if(is.null(dim(x))==T){
          window(filtered,end=sx)<-NA
          window(filtered,start=ex)<-NA 
        }else{
          for (i in 1:dim(x)[2]){
            window(filtered[,i],end=sx[i])<-NA
            window(filtered[,i],start=ex[i])<-NA
          }
        }
      }
    }
  }
  if(method=="gauss"){
    if(is.null(dim(x))==T){
      filtered<-gauss.na(x,width)
    }else{ 
      filtered<-ts(apply(x,2,function(y) gauss.na(y,width)),start=start(x)[1])
    }
  }
  if(method=="rm"){
    if(is.null(dim(x))==T){
      filtered<-rollmean.na(x,width)
    }else{ 
      if(length(which(is.na(x)))>0){
        filtered<-ts(apply(x,2,function(y) rollmean.na(y,width)),start=start(x)[1])
      }else{
        filtered<-rollmean(x,width)
      }
    } 
  }
  if(method=="hamming"){
    require(oce)
    filtered<-stats::filter(x,makeFilter("hamming", 50, asKernel=FALSE))
  }
  
  
  if(method=="bw"){
    if(is.null(dim(x))==T){
      filtered<-butterfilt.na(x,width)
    }else{ 
      filtered<-tsapply(x,2,function(y) butterfilt.na(y,width))
    }
    if(cut.end==T){
      sx<-start(x)[1]+floor(width/2)-1
      ex<-end(x)[1]-floor((width-0.5)/2)+1
      window(filtered,end=sx)<-NA
      window(filtered,start=ex)<-NA
    }
  }
  
  
  filtered
}


butterfilt.na<-function(y,tsc,type="low",order=2){
  #require(signal)
  # dt is expected to be equal to 1 sampling unit!!
  #first get rid of NAs at begining and nend
  sy<-min(which(!is.na(y)))
  ey<-max(which(!is.na(y)))
  x<-y[sy:ey]
  mx<-mean(x)
  #anomalies to full peroid mean may help to reduce end effects in some cases
  x<-x-mx
  nx<-tsc*order
  x2<-c(rep(mean(x[1:tsc]),nx),x,rep(mean(x[(length(x)-tsc+1):length(x)]),nx))
  bf <- signal:::butter(order, 1/tsc, type=type)
  b1 <- signal:::filtfilt(bf, x2)
  b1 <- b1[-c(1:nx,(length(b1)-nx+1):length(b1))]
  #pl.mts(cbind(x2,b1))
  b1<-b1+mx
  z<-y
  z[sy:ey]<-b1
  return(z)
  #detach(package:signal,unload=TRUE,force=TRUE)
  #unloadNamespace("signal")
  #  freqz(b1)
  #  zplane(bf)
}


anomalies.period<-function(data,start,end){
  if(is.null(dim(data))==T){
    data<-data-mean(window(data,start=start,end=end),na.rm=T)
  }else{ 
    sy<-which(time(data)==start)
    ey<-which(time(data)==end)
    data<-ts(apply(data,2,function(x) x-mean(x[sy:ey],na.rm=T)),start=start(data)[1])
  }
  return(data)
}

#combine elements of x into a matrix (as columns)
#only works if all elements are of same length or ts.
cbind.list<-function(x){
  matrix<-1
  for(i in 1:length(x)){
    matrix<-cbind(matrix,x[[i]])
  }
  matrix<-matrix[,-1]
  matrix
}

generate.rmvn.ar<-function(x,use="everything",n=NA){
  library(mgcv)
  #library(Rfast)  
  #covs<-cov(x,use=use)
  #covs<-cova(x) #is faster than cov
  covs<-cov2(x) #is faster than cova for large matrices
  if(is.na(n)) n<-dim(x)[1]
  acs<-apply(x,2,fastacf) 
  
  yerr<-mgcv::rmvn(n=n, mu=rep(0,dim(x)[2]),V = covs)
  for(i in 1:dim(x)[2]){
    noise<-yerr[1,i]
    for(t in 2:n){
      noise[t]<-acs[i]*noise[t-1]+yerr[t,i]
    }
    sdf<-sqrt(var(x[,i]))/sqrt(var(noise))
    noise<-noise*sdf
    mf<-mean(x[,i])-mean(noise)
    yerr[,i]<-noise+mf
  }
  yerr
}


##fast version of cov
#one may also use cova() from the Rfast package which is also faster than cov but cov2 appears to be faster for large matrices
cov2 <- function(x){
  1/(NROW(x) -1) * crossprod(scale(x, TRUE , FALSE))
}


fastacf<-function(x){
  if(is.null(dim(x))){
    ac<-cor(x,c(x[-1],NA),use="complete.obs")
  }else{
    ac<-apply(x,2,function(y) cor(y,c(y[-1],NA),use="complete.obs"))
  }
  ac
}

#make boxplots using quantile-based whiskers instead of fractions of the interquartile range. end of whiskers will be at quantiles defined by min and max.
#call bxp.q to allow for multiple datasets; plot via bxp()
boxplot.quantiles<-function(x,min=0.1,max=0.9){
  bx<-boxplot(x,plot=F)
  if(is.list(x)==T){
    minx<-unlist(lapply(x,function(x) quantile(x,probs=min,na.rm=T)))
    maxx<-unlist(lapply(x,function(x) quantile(x,probs=max,na.rm=T)))
    bx$stats[1,]<-minx
    bx$stats[5,]<-maxx
  }else{
    minx<-quantile(x,probs=min,na.rm=T)
    maxx<-quantile(x,probs=max,na.rm=T) 
    bx$stats[1]<-minx
    bx$stats[5]<-maxx
  }
  
  bx$stats[1]<-minx
  bx$stats[5]<-maxx
  if(is.list(x)==T){
    bx$out<-unlist(lapply(x,function(x) x[which(x<minx | x>maxx)]))
  }else{
    bx$out<-x[which(x<minx | x>maxx)]
  }
  bx$group<-rep(1,length(bx$out))
  return(bx)
}

bxp.q<-function(x,min=0.1,max=0.9){
  if(is.list(x)==T){
    
    for(n in 1:length(x)){
      bx<-boxplot.quantiles(x[[n]],min,max)
      if(n==1){
        out<-bx
      }else{
        out$stats<-cbind(out$stats,bx$stats)
        out$n<-c(out$n,bx$n)
        out$conf<-cbind(out$conf,bx$conf)
        out$out<-c(out$out,bx$out)
        out$group<-c(out$group,rep(n,length(bx$out)))
        out$names<-c(out$names,n)
      }
    }
  }else{
    if(length(dim(x)[2])>0){
      
      for(n in 1:dim(x)[2]){
        bx<-boxplot.quantiles(x[,n],min,max)
        if(n==1){
          out<-bx
        }else{
          out$stats<-cbind(out$stats,bx$stats)
          out$n<-c(out$n,bx$n)
          out$conf<-cbind(out$conf,bx$conf)
          out$out<-c(out$out,bx$out)
          out$group<-c(out$group,rep(n,length(bx$out)))
          out$names<-c(out$names,n)
        }
      }
    }else{
      out<-boxplot.quantiles(x,min,max)
    }
  }
  return(out)
}

# function to make the text width for each element of the legend correct if horiz=T, leg is the legend text e.g. leg<-c("line1","line2")
#use text.width=tw1 in the legend call
twf<-function(leg,cex=1){
  x<-strwidth(leg[-length(leg)],cex=cex)
  tw1<-c(0,cumsum(x)/(1:length(x)))
}


#Tol" palette for colorblind
colblind.cols.tol<-c("#332288","#117733","#44AA99","#88CCEE","#DDCC77","#CC6677","#AA4499","#882255")
