# This script assumes that each XML element tag stands alone on a line.
# That is, the beginning of an tag may only be preceded by whitespace, and the
# end of an element may only be followed by whitespace.  An element tag may span
# several lines, but multiple tags (or partial tags) are not allowed to share a
# line, and no non-tag characters may share a line with a tag.


# Get the attributes of the element and store them into the Attributes array
# with attribute values indexed by attribute names.
#
function getElementAttributes(sElement, asAttributes, \
                              asNames, asValues, nFields, sTmp1, sTmp2, key) # local variables
{
    # Initialize the local variables.
    #
    nFields = 0;
    sTmp1   = "";
    sTmp2   = "";
    key     = "";

    delete asNames;
    delete asValues;

    # Clear the output asAttributes array.
    #
    delete asAttributes;

    # Strip away the pieces of the element that are not the element attribute
    # name=value pairs.  Then, delimit each pair with tabs.  There will be a
    # leading tab, so remove it.
    #
    sTmp1 = gensub(/^[[:space:]]*<[[:space:]]*\w+(([[:space:]]+\w+[[:space:]]*=[[:space:]]*"[^"]*")+)[[:space:]]*\/?>[[:space:]]*$/, "\\1", "1", sElement);
    sTmp1 = gensub(/[[:space:]]+(\w+[[:space:]]*=[[:space:]]*"[^"]*")/, "\t\\1", "g", sTmp1);
    sTmp1 = gensub(/^\t(.*)$/, "\\1", "1", sTmp1);

    # Remove the '=value' parts of the name=value pairs, so that the result is a
    # tab-delimited list of names.  Split the string into the Names array.
    #
    sTmp2 = gensub(/(\w+)[[:space:]]*=[[:space:]]*"[^"]*"/, "\\1", "g", sTmp1);

    split(sTmp2, asNames, "\t");

    # Remove the 'name=' parts of the name=value pairs, so that the result is a
    # tab-delimited list of values.  Split the string into the Values array.
    #
    sTmp2 = gensub(/\w+[[:space:]]*=[[:space:]]*"([^"]*)"/, "\\1", "g", sTmp1);

    nFields = split(sTmp2, asValues, "\t");

    # Store the values into the argument Attributes array indexed by the names
    # for the values.
    #
    for (key = 1 ; key <= nFields ; key++)
    {
        asAttributes[asNames[key]] = asValues[key];
    }
}


# Process an attribute element.
#
function processAttributeElement( \
                                 nI, nJ, elem) # local variables
{
    # If the attribute is part of a variable attribute, add the information to
    # the variable attributes arrays, otherwise, add the information to the
    # group attributes arrays.
    #
    if (1 == nIsInVariable)
    {
        # Get the count of variables for the current group.
        #
        nI = anGroupNumVariables[sGroupPath];

        # Get and increment the number of variable attributes.
        #
        nJ = ++anGroupVariableNumAttributes[sGroupPath,nI];

        # Store the information for the attribute.
        #
        for (elem in asElements)
        {
            asGroupVariableAttributes[sGroupPath,nI,nJ,elem] = asElements[elem];
        }
    }
    else
    {
        # Get and increment the number of group attributes.
        #
        nI = ++anGroupNumAttributes[sGroupPath];

        # Store the information for the attribute.
        #
        for (elem in asElements)
        {
            asGroupAttributes[sGroupPath,nI,elem] = asElements[elem];
        }
    }
}


function processValuesElement(asElements, sValues, \
                              nI, nJ, sSep, elem, key) # local variables
{
    # If the Values argument is empty, check to see if the Elements array
    # contains a start/increment pair and handle it.  If not, return without
    # doing anything.  If the Values argument is not empty, handle it.
    #
    if ("" == sValues)
    {
        if (!("start" in asElements) || !("increment" in asElements))
        {
            return;
        }

        # Create a Values string specifying the start and increment.
        #
        sValues = "start=" asElements["start"] " increment=" asElements["increment"];
    }
    else
    {
        # Strip off any leading or trailing whitespace.  Turn any runs of one or
        # more whitespace characters into single spaces.
        #
        sValues = gensub(/^[[:space:]]+/, "", "1", sValues);
        sValues = gensub(/[[:space:]]+$/, "", "1", sValues);
        sValues = gensub(/[[:space:]]+/, " ", "g", sValues);
    }

    # Add the information to the variable attributes arrays.
    #
    # Get the count of variables for the current group.
    #
    nI = anGroupNumVariables[sGroupPath];

    # Get and increment the number of variable attributes.
    #
    nJ = ++anGroupVariableNumAttributes[sGroupPath,nI];

    # Store the information for the values.
    #
    asGroupVariableAttributes[sGroupPath,nI,nJ,"name"]  = "values";
    asGroupVariableAttributes[sGroupPath,nI,nJ,"type"]  = asGroupVariables[sGroupPath,nI,"type"];
    asGroupVariableAttributes[sGroupPath,nI,nJ,"value"] = sValues;
}


# Process a variable element.
#
function processVariableElement( \
                                nI, nJ, elem) # local variables
{
    # Get and increment the number of group variables.
    #
    nI = ++anGroupNumVariables[sGroupPath];

    # Store the information for the variable.
    #
    for (elem in asElements)
    {
        asGroupVariables[sGroupPath,nI,elem] = asElements[elem];
    }

    # Initialize the variable attributes count array element.
    #
    anGroupVariableNumAttributes[sGroupPath,nI] = 0;
}


# Initialize global variables.
#
BEGIN \
{
    nIsInAttribute = 0;
    nIsInValues    = 0;
    nIsInVariable  = 0;

    sAttributeValue = "";
    sValues         = "";

    # Label the root group as "Global".
    #
    sGroupPath = "Global";
    
    nNumGroups = 1;

    asGroups[1] = "Global";

    anGroupNumAttributes["Global"] = 0;
    anGroupNumDimensions["Global"] = 0;
    anGroupNumVariables["Global"]  = 0;

    asGroupAttributes["Global",1,"name"] = "";
    asGroupDimensions["Global",1,"name"] = "";
    asGroupVariables["Global",1,"name"]  = "";

    anGroupVariableNumAttributes["Global",1] = 0;

    asGroupVariableAttributes["Global",1,1,"name"] = "";

    asElements["dummy"] = "";
}


#Skip empty lines.
#
NF == 0 \
{
    next;
}


# If a line contains the start of a tag with no corresponding end, concatenate
# lines until the tag is complete.
#
$0 ~ /<[^>]*$/ \
{
    do
    {
        getline sTmp;

        $0 = $0 " " sTmp;
    }
    while (sTmp !~ /^[^>]*>/);
}


# Process a group element start tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*group[[:space:]]+/ \
{
    # Get the element attributes for the group.
    #
    getElementAttributes($0, asElements);

    # Increment the group count.
    #
    nNumGroups++;

    # Add the name of the group to the path, then store the path in the Groups
    # array.
    #
    sGroupPath = sGroupPath "/" asElements["name"];

    asGroups[nNumGroups] = sGroupPath;

    # Initialize the counting arrays for the new group.
    #
    anGroupNumAttributes[sGroupPath] = 0;
    anGroupNumDimensions[sGroupPath] = 0;
    anGroupNumVariables[sGroupPath]  = 0;

    # Move to the next line.
    #
    next;
}


# Process a group element end tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*\/group[[:space:]]*>/ \
{
    #Remove the last element from the group path.
    #
    sGroupPath = gensub(/\/[^\/]+$/, "", "1", sGroupPath);

    # Move to the next line.
    #
    next;
}


# Process a dimension element.
#
$0 ~ /^[[:space:]]*<[[:space:]]*dimension[[:space:]]+/ \
{
    # Get the element attributes.
    #
    getElementAttributes($0, asElements);

    # Get and increment the number of dimensions for the group.
    #
    nI = ++anGroupNumDimensions[sGroupPath];

    # Store the information for the dimension.
    #
    for (elem in asElements)
    {
        asGroupDimensions[sGroupPath,nI,elem] = asElements[elem];
    }

    # Move to the next line.
    #
    next;
}


# Process a variable element. (This one shouldn't be seen... but just in case.)
#
$0 ~ /^[[:space:]]*<[[:space:]]*variable[[:space:]]+.*\/>/ \
{
    # Get the element attributes.
    #
    getElementAttributes($0, asElements);

    # Do the work.
    #
    processVariableElement();

    # Move to the next line.
    #
    next;
}


# Process a variable element start tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*variable[[:space:]]+[^\/>]*>/ \
{
    # Get the element attributes.
    #
    getElementAttributes($0, asElements);

    # Do the work.
    #
    processVariableElement();

    # Set the IsInVariable flag to true.
    #
    nIsInVariable = 1;

    # Move to the next line.
    #
    next;
}


# Process a variable element end tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*\/variable[[:space:]]*>/ \
{
    #Set the IsInVariable flag to false.
    #
    nIsInVariable = 0;

    # Move to the next line.
    #
    next;
}


# Process an attribute element.
#
$0 ~ /^[[:space:]]*<[[:space:]]*attribute[[:space:]]+.*\/>/ \
{
    # Get the element attributes.
    #
    getElementAttributes($0, asElements);

    # Do the work.
    #
    processAttributeElement();

    # Move to the next line.
    #
    next;
}


# Process an attribute element start tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*attribute[[:space:]]+.*[^\/]>/ \
{
    getElementAttributes($0, asElements);

    # Set the IsInAttribute flag to true.
    #
    nIsInAttribute = 1;

    # Clear the sAttributeValue string.
    #
    sAttributeValue = "";

    # Move to the next line.
    #
    next;
}


# Process an attribute element end tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*\/attribute[[:space:]]*>/ \
{
    # if it isn't empty, add the value to the element attributes.
    #
    if (sAttributeValue != "")
    {
        # Strip off any leading or trailing whitespace.  Turn any runs of one or
        # more whitespace characters into single spaces.
        #
        sTmp = gensub(/^[[:space:]]+/, "", "1", sAttributeValue);
        sTmp = gensub(/[[:space:]]+$/, "", "1", sTmp);
        sTmp = gensub(/[[:space:]]+/, " ", "g", sTmp);
        
        # Store the cleaned-up string into the "value" element of the Elements
        # array.
        #
        asElements["value"] = sTmp;
    }

    # Do the work.
    #
    processAttributeElement();

    #Set the IsInAttribute flag to false.
    #
    nIsInAttribute = 0;

    # Move to the next line.
    #
    next;
}


# Process an attribute element value line.
#
nIsInAttribute == 1 \
{
    # Append the current line to the attribute value string, inserting a space
    # between the existing string and the new contents.
    #
    if (sAttributeValue != "")
    {
        sAttributeValue = sAttributeValue " ";
    }

    sAttributeValue = sAttributeValue $0;

    # Move to the next line.
    #
    next;
}


# Process a values element tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*values[[:space:]]+[^\/]+\/>/ \
{
    getElementAttributes($0, asElements);

    # Add the value to the variable element attributes.
    #
    processValuesElement(asElements, "");

    # Move to the next line.
    #
    next;
}


# Process a values element start tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*values[[:space:]]+.*[^\/]>/ \
{
    getElementAttributes($0, asElements);

    # Set the IsInValues flag to true.
    #
    nIsInValues = 1;

    # Clear the sValues string.
    #
    sValues = "";

    # Move to the next line.
    #
    next;
}


# Process a values element end tag.
#
$0 ~ /^[[:space:]]*<[[:space:]]*\/values[[:space:]]*>/ \
{
    # if it isn't empty, add the value to the variable element attributes.
    #
    if (sValues != "")
    {
        processValuesElement(asElements, sValues);
    }

    #Set the IsInValues flag to false.
    #
    nIsInValues = 0;

    # Move to the next line.
    #
    next;
}


# Process a values element value line.
#
nIsInValues == 1 \
{
    # Append the current line to the values string, inserting a space
    # between the existing string and the new contents.
    #
    if (sValues != "")
    {
        sValues = sValues " ";
    }

    sValues = sValues $0;

    # Move to the next line.
    #
    next;
}


# The file has been parsed.  Call the output function.  This function is defined
# in a different script file.
END \
{
    generateOutput();
}
