# The functions in this script assume the existence of a number of global
# variables.  They are:
#   nNumGroups                   The number of groups parsed
#   asGroups                     Array of group paths
#   anGroupNumAttributes         Array of the number of attributes per group
#   anGroupNumDimensions         Array of the number of dimensions per group
#   anGroupNumVariables          Array of the number of variables per group
#   asGroupAttributes            Array of info about each group attribute
#   asGroupDimensions            Array of info about each dimension
#   asGroupVariables             Array of info about each variable
#   anGroupVariableNumAttributes Array of the number of attributes per variable
#   asGroupVariableAttributes    Array of info about each variable attribute
#   asDimensionDictionary        Array of dimension names indexed by group path


# This function is called by the CRDR_NcMLtoOutput awk script to generate output
# once the input NcML file has been parsed.  It generates a TSV file that can be
# used as input to the CRDR_TSVtoNcML script.
#
function generateOutput( \
                        asFillValueTable, asMissingValueTable, nI, sGroupPath) # local variables
{
    # Initialize local variables that need it.
    #
    delete asFillValueTable;
    delete asMissingValueTable;

    # Handle variable and attribute elements that lack a type specifier.
    #
    processMissingTypes();

    # Convert XML character escape sequences in string values into regular
    # characters.
    #
    processXmlCharacters();

    # Handle scaled variables.
    #
    processScaledVariables();

    # Add "n/a" values for variables that don't have _FillValue or missing_value
    # attributes.
    #
    addNotApplicableValues();

    # If collection of coordinates values that specify the shape coordinates
    # was requested, trim the entries in the variable attributes.
    #
    if (1 == nCollectCoordinatesValues)
    {
        collectCoordinatesValues();
    }

    # If collection of fill values was requested, build the table and remove
    # entries from the variable attributes.
    #
    if (1 == nCollectFillValues)
    {
        collectFillValues(asFillValueTable);
    }

    # If collection of full-range valid_min/max values was requested, remove
    # entries from the variable attributes.
    #
    if (1 == nCollectMinMaxValues)
    {
        collectMinMaxValues();
    }

    # If collection of missing_value values was requested, build the table and
    # remove entries from the variable attributes.
    #
    if (1 == nCollectMissingValues)
    {
        collectMissingValues(asMissingValueTable);
    }

    # Write the preamble.
    #
    generateFilePreamble(asFillValueTable, asMissingValueTable);

    # Generate output for each group.
    #
    for (nI = 1 ; nI <= nNumGroups ; nI++)
    {
        sGroupPath = asGroups[nI];

        generateGroupOutput(sGroupPath);
    }
}


# Step through all of the attributes, variables, and variable attributes
# looking for elements with missing type specifications.  Add default types
# where needed.
#
function processMissingTypes( \
                             nAtt, nAtts, nGroup, nVar, nVars, sGroupPath, sType, sVarType) # local variables
{
    # Examine each attribute and variable in each group to see if it is missing
    # a type specifier.  If it is, specify an appropriate default.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        # Get the group path.
        #
        sGroupPath = asGroups[nGroup];

        # If a group attribute is missing a type specifier, give it a type of
        # "string".
        #
        nAtts = anGroupNumAttributes[sGroupPath];
        
        for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
        {
            if (!asGroupAttributes[sGroupPath,nAtt,"type"])
            {
                asGroupAttributes[sGroupPath,nAtt,"type"] = "string";
            }
        }

        # If a group variable is missing a type specifier, give it a type of
        # "string", then examine the variable attributes.
        #
        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            sVarType = asGroupVariables[sGroupPath,nVar,"type"];

            if (!sVarType)
            {
                asGroupVariables[sGroupPath,nVar,"type"] = "string";

                sVarType = "string";
            }

            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # If a variable attribute is missing a type specifier, give it a
            # type of "string".
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if (!asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"])
                {
                    asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"] = "string";
                }
            }
        }
    }
}


# Step through all of the attributes, variables, and variable attributes
# looking for elements of string type.  Convert any XML character escape
# sequences into regular characters in the values for those elements.
#
function processXmlCharacters( \
                             nAtt, nAtts, nGroup, nVar, nVars, sGroupPath, sVarType) # local variables
{
    # Examine each attribute and variable in each group to see if it has a type
    # of "string".  If it does, handle any XML character escape sequences found
    # in the value.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        # Get the group path.
        #
        sGroupPath = asGroups[nGroup];

        # If a group attribute has a type of "string", process it.
        #
        nAtts = anGroupNumAttributes[sGroupPath];
        
        for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
        {
            if ("string" == asGroupAttributes[sGroupPath,nAtt,"type"])
            {
                asGroupAttributes[sGroupPath,nAtt,"value"] = convertFromXmlCharacters(asGroupAttributes[sGroupPath,nAtt,"value"]);
            }
        }

        # If a group variable has a type of "string", process it.
        #
        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            sVarType == asGroupVariables[sGroupPath,nVar,"type"];

            if ("string" == sVarType)
            {
                asGroupVariables[sGroupPath,nVar,"value"] = convertFromXmlCharacters(asGroupVariables[sGroupPath,nVar,"value"]);
            }

            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # If a variable attribute has a type of "string", or if it has a
            # type of "type" and the variable has a type of "string", process
            # it.
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                sType = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];

                if ("type" == sType)
                {
                    sType = sVarType;
                }

                if ("string" == sType)
                {
                    asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"] = convertFromXmlCharacters(asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"]);
                }
            }
        }
    }
}


# Process variables that are "scaled".  (That is, that have "add_offset" and/or
# "scale_factor" attributes.) For each scaled variable, create a new
# pseudo-attribute named "scaled_type", and change the type of the "add_offset"
# and "scale_factor" attributes to "scaled_type".
#
function processScaledVariables( \
                                nAtt, nAtts, nGroup, nOffsetIndex, nScaleIndex, nVar, nVars, sGroupPath, sName, sType) # local variables
{
    # Examine each variable in each group to see if it has a "add_offset" or
    # "scale_factor" attribute.  If it does, modify it.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # If there is a "add_offset" attribute or "scale_factor" attribute,
            # capture the index and the type.  (The type should be the same for
            # both.)  Change the type for each attribute to "type".
            #
            nOffsetIndex = 0;
            nScaleIndex  = 0;
            sType        = "";

            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                sName = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];

                if ("add_offset" == sName)
                {
                    sType = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];

                    asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"] = "type";

                    nOffsetIndex = nAtt;
                }
                else if ("scale_factor" == sName)
                {
                    sType = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];

                    asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"] = "type";

                    nScaleIndex = nAtt;
                }
            }

            # If neither the offset or scale attributes was found, skip to the
            # next variable.
            #
            if (0 == nOffsetIndex && 0 == nScaleIndex)
            {
                continue;
            }

            # Add an attribute named "scaled_type", with a value of the scaled type.
            #
            nAtts++;

            anGroupVariableNumAttributes[sGroupPath,nVar]++;

            asGroupVariableAttributes[sGroupPath,nVar,nAtts,"name"]  = "scaled_type";
            asGroupVariableAttributes[sGroupPath,nVar,nAtts,"type"]  = "string";
            asGroupVariableAttributes[sGroupPath,nVar,nAtts,"value"] = sType;
        }
    }
}


# Add variable attribute entries for _FillValue and missing_value with values
# of "n/a" to variables that don't have these attributes.
#
function addNotApplicableValues( \
                                anVariableIsCoordinate, nAtt, nAtts, nGroup, nNoFill, nNoMissing, nVar, nVars, sGroupPath, sName, sType) # local variables
{
    # Initialize the local variables that need it.
    #
    delete anVariableIsCoordinate;
    
    # Examine each variable in each group to see if it has a _FillValue or
    # missing_value attribute.  If it doesn't, add one with a value of "n/a".
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        # Determine which variables are coordinates.
        #
        createVariableIsCoordinateArray(sGroupPath, anVariableIsCoordinate);

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # If the variable is a coordinate, skip it.
            #
            if (1 == anVariableIsCoordinate[nVar])
            {
                continue;
            }

            # Get the variable type.
            #
            sType = asGroupVariables[sGroupPath,nVar,"type"];

            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # If there is no _FillValue attribute or missing_value attribute,
            # set the appropriate flag.
            #
            nNoFill = 1;
            nNoMissing = 1;

            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                sName = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];

                if ("_FillValue" == sName)
                {
                    nNoFill = 0;
                }
                else if ("missing_value" == sName)
                {
                    nNoMissing = 0;
                }
            }

            # If there was no _FillValue attribute, add one with a value of
            # "n/a".  Increment the count of attributes for the variable.
            #
            if (1 == nNoFill)
            {
                nAtts++;

                anGroupVariableNumAttributes[sGroupPath,nVar]++;

                asGroupVariableAttributes[sGroupPath,nVar,nAtts,"name"]  = "_FillValue";
                asGroupVariableAttributes[sGroupPath,nVar,nAtts,"type"]  = sType;
                asGroupVariableAttributes[sGroupPath,nVar,nAtts,"value"] = "n/a";
            }

            # If there was no missing_value attribute, add one with a value of
            # "n/a".  Increment the count of attributes for the variable.
            #
            if (1 == nNoMissing)
            {
                nAtts++;

                anGroupVariableNumAttributes[sGroupPath,nVar]++;

                asGroupVariableAttributes[sGroupPath,nVar,nAtts,"name"]  = "missing_value";
                asGroupVariableAttributes[sGroupPath,nVar,nAtts,"type"]  = sType;
                asGroupVariableAttributes[sGroupPath,nVar,nAtts,"value"] = "n/a";
            }
        }
    }
}


# Remove the parts of coordinates attributes from variables where the attribute
# value matches the full-scale value for the variable type.
#
function collectCoordinatesValues( \
                                  asParts, anCoords, anShapeCoords, nAtt, nAtts, nGroup, nVar, nVars, sGroupPath, \
                                  sCoord, sShape, sCoordinates, nCoord, nCoords, key) # local variables
{
    # Initialize the local variables that need it.
    #
    delete asParts;
    delete anCoords;
    delete anShapeCoords;

    # Examine each variable in each group to see if it has a coordinates
    # attribute.  If it does, handle it.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # Get the variable shape.  If the shape is empty, move to the next
            # variable.
            #
            sShape = asGroupVariables[sGroupPath,nVar,"shape"];

            if (!sShape)
            {
                continue;
            }

            # Build a list of the coordinate names in the shape.
            #
            split(sShape, asParts, / +/);

            for (key in asParts)
            {
                # If there is a variable with the name of the dimension from
                # the shape, add the key to the ShapeCoords array indexed by
                # the coordinate name.
                #
                sCoord = asParts[key];

                for (nCoord = 1 ; nCoord <= nVars ; nCoord++)
                {
                    if (sCoord == asGroupVariables[sGroupPath,nCoord,"name"])
                    {
                        anShapeCoords[sCoord] = key;

                        break;
                    }
                }
            }

            # If there is a "coordinates" attribute, process it.
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("coordinates" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    # Build a list of the coordinate names.
                    #
                    sValue = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

                    split(sValue, asParts, / +/);

                    for (key in asParts)
                    {
                        anCoords[asParts[key]] = key;
                    }

                    # Remove the shape coordinates from the list of
                    # coordinates.
                    #
                    for (key in anShapeCoords)
                    {
                        if (key in anCoords)
                        {
                            delete anCoords[key];
                        }
                    }

                    # Count the number of coordinates that remain.  If the
                    # number is zero, delete the coordinates attribute.
                    # Otherwise, build a string of the coordinate names that
                    # remain and store it back into the attribute value.
                    #
                    nCoords = 0;

                    for (key in anCoords)
                    {
                        nCoords++;
                    }

                    if (0 == nCoords)
                    {
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];
                    }
                    else
                    {
                        # Invert the anCoords array.  Sort the coordinates so
                        # that they are in original appearance order.
                        #
                        delete asParts;

                        for (key in anCoords)
                        {
                            asParts[anCoords[key]] = key;
                        }

                        nCoords = asort(asParts);

                        # Build the coordinates string.
                        #
                        sCoordinates = asParts[1];

                        for (key = 2 ; key <= nCoords ; key++)
                        {
                            sCoordinates = sCoordinates " " asParts[key];
                        }

                        # Store the new value back into the attribute.
                        #
                        asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"] = sCoordinates;
                    }

                    break;
                }
            }
        }
    }
}


# Build a table of fill values based on the most-used values for each variable
# type.  Once the table is built, remove _FillValue attributes from variables
# where the attribute value matches the table entry for the variable type.
# Return the fill value table in the argument array.
#
function collectFillValues(asFillValueTable, \
                           anFillMatch, anFillValueCounts, asFillMax, asParts, nAtt, nAtts, nCount, nDiff, nGroup, nVar, nVars, \
                           sGroupPath, sValue, sVarType, key) # local variables
{
    # Initialize the local variables that need it.
    #
    delete anFillMatch;
    delete anFillValueCounts;
    delete asFillMax;

    # Clear the FillValueTable array.
    #
    delete asFillValueTable;

    # Collect counts of the values of the _FillValue attributes.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # Get the variable type.  If the variable has an attribute named
            # "_Unsigned" that has a value of "true", make the variable type
            # unsigned.
            #
            sVarType = asGroupVariables[sGroupPath,nVar,"type"];

            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("_Unsigned" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    if ("true" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"])
                    {
                        sVarType = "unsigned " sVarType;
                    }

                    break;
                }
            }

            # If there is a "_FillValue" attribute that has a value other than
            # "n/a", add it to the collection of counts, indexed by the
            # variable type.
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("_FillValue" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    sValue = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

                    if ("n/a" != sValue)
                    {
                        anFillValueCounts[sVarType,sValue]++;
                    }

                    break;
                }
            }
        }
    }

    # For each type, find the fill value that has the most instances.
    #
    for (key in anFillValueCounts)
    {
        # Get the type and value indices for the key.
        #
        split(key, asParts, SUBSEP);

        sVarType = asParts[1];
        sValue   = asParts[2];

        # Get the count for the type and value.
        #
        nCount = anFillValueCounts[key];

        # If the count is greater than the current max count for the type, save
        # the key for the new max and set the number of matching counts to 1.
        # If the count matches the current max count, increment the number of
        # matching counts.
        #
        nDiff = nCount - anFillValueCounts[asFillMax[sVarType]];

        if (0 < nDiff)
        {
            asFillMax[sVarType]   = key;
            anFillMatch[sVarType] = 1;
        }
        else if (0 == nDiff)
        {
            anFillMatch[sVarType]++;
        }
    }

    # For each type that has a fill value with more instances than any other
    # fill value, store the value into the asFillValueTable indexed by type.
    #
    for (sVarType in anFillMatch)
    {
        if (1 == anFillMatch[sVarType])
        {
            # Get the fill value from the key stored in the FillMax array.  It
            # is the second index.
            #
            split(asFillMax[sVarType], asParts, SUBSEP);

            asFillValueTable[sVarType] = asParts[2];
        }
    }

    # Examine each "_FillValue" attribute in each variable in each group.  If
    # it matches the entry for the variable type in the fill value table,
    # delete the attribute.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # Get the variable type.  If the variable has an attribute named
            # "_Unsigned" that has a value of "true", make the variable type
            # unsigned.
            #
            sVarType = asGroupVariables[sGroupPath,nVar,"type"];

            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("_Unsigned" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    if ("true" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"])
                    {
                        sVarType = "unsigned " sVarType;
                    }

                    break;
                }
            }

            # If there is a "_FillValue" attribute, compare the value against
            # the fill value table.  If there is a match, delete the attribute.
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("_FillValue" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    sValue = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

                    if ((sVarType in asFillValueTable) && (sValue == asFillValueTable[sVarType]))
                    {
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];
                    }

                    break;
                }
            }
        }
    }
}


# Remove valid_min and valid_max attributes from variables where the attribute
# value matches the full-scale value for the variable type.
#
function collectMinMaxValues( \
                             asValidMaxByType, asValidMinByType, nAtt, nAtts, nGroup, nVar, nVars, sGroupPath, sValue, sVarType) # local variables
{
    # Initialize the local variables that need it.
    #
    delete asValidMaxByType;
    delete asValidMinByType;

    asValidMaxByType["byte"]           = "127";
    asValidMaxByType["unsigned byte"]  = "255";
    asValidMaxByType["short"]          = "32767";
    asValidMaxByType["unsigned short"] = "65535";
    asValidMaxByType["int"]            = "2147483647";
    asValidMaxByType["unsigned int"]   = "4294967295";
    asValidMaxByType["long"]           = "9223372036854775807";
    asValidMaxByType["unsigned long"]  = "18446744073709551615";
    asValidMaxByType["float"]          = "3.40282347e+38";
    asValidMaxByType["double"]         = "1.7976931348623157e+308";

    asValidMinByType["byte"]           = "-128";
    asValidMinByType["unsigned byte"]  = "0";
    asValidMinByType["short"]          = "-32768";
    asValidMinByType["unsigned short"] = "0";
    asValidMinByType["int"]            = "-2147483648";
    asValidMinByType["unsigned int"]   = "0";
    asValidMinByType["long"]           = "-9223372036854775808";
    asValidMinByType["unsigned long"]  = "0";
    asValidMinByType["float"]          = "-3.40282347e+38";
    asValidMinByType["double"]         = "-1.7976931348623157e+308";

    # Examine each variable in each group to see if it has a valid_min or
    # valid_max attribute with a full-range value.  If it does, delete it.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # Get the variable type.  If the variable has an attribute named
            # "_Unsigned" that has a value of "true", make the variable type
            # unsigned.
            #
            sVarType = asGroupVariables[sGroupPath,nVar,"type"];

            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("_Unsigned" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    if ("true" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"])
                    {
                        sVarType = "unsigned " sVarType;
                    }

                    break;
                }
            }

            # If there is a "valid_min" or "valid_max" attribute, check it
            # against the full-range value for the type.  If there is a match,
            # delete the attribute.
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("valid_max" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    sValue = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

                    if (sValue == asValidMaxByType[sVarType])
                    {
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];
                    }
                }
                else if ("valid_min" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    sValue = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

                    if (sValue == asValidMinByType[sVarType])
                    {
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];
                    }
                }
            }
        }
    }
}


# Build a table of missing values based on global "missing_value_meanings" and
# "missing_values_<n>bit_[un]signed" attributes.  Once the table is built,
# remove the global attributes used to build the table, and remove missing
# value attributes from variables where the attribute value matches the table
# entry for the variable type.  Return the missing value table in the argument
# array.
#
function collectMissingValues(asMissingValueTable, \
                              asParts, asTypes, nAtt, nAtts, nCount, nDiff, nGroup, nPart, nParts, nVar, nVars, \
                              sGroupPath, sName, sValue, sVarType, key) # local variables
{
    # Initialize local variables that need it.
    #
    delete asTypes;

    asTypes["8","signed"]    = "byte";
    asTypes["8","unsigned"]  = "unsigned byte";
    asTypes["16","signed"]   = "short";
    asTypes["16","unsigned"] = "unsigned short";
    asTypes["32","float"]    = "float";
    asTypes["32","signed"]   = "int";
    asTypes["32","unsigned"] = "unsigned int";
    asTypes["64","float"]    = "double";
    asTypes["64","signed"]   = "long";
    asTypes["64","unsigned"] = "unsigned long";

    # Clear the MissingValueTable array.
    #
    delete asMissingValueTable;

    # If a missing value table is encoded in the global attributes, use the
    # attributes to build the contents of the MissingValueTable array.  Delete
    # the attributes once they have been used.
    #
    nAtts = anGroupNumAttributes["Global"];

    for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
    {
        # If the attribute name is "missing_value_meanings", store the meanings
        # in consecutive elements of the table, with a first index of
        # "meaning".
        #
        sName = asGroupAttributes["Global",nAtt,"name"];

        if ("missing_value_meanings" == sName)
        {
            # The meanings value is a string of names separated by spaces.
            # Split the string into the names.
            #
            nVars = split(asGroupAttributes["Global",nAtt,"value"], asParts, / +/);

            # Store the meanings into the table.
            #
            for (nVar = 1 ; nVar <= nVars ; nVar++)
            {
                asMissingValueTable["meaning",nVar] = asParts[nVar];
            }

            delete asGroupAttributes["Global",nAtt,"name"];
            delete asGroupAttributes["Global",nAtt,"type"];
            delete asGroupAttributes["Global",nAtt,"value"];

            continue;
        }

        # If the attribute name is
        # "missing_value_<n8|16|32|64>bit_<signed|unsigned|float|>", the value
        # is a string of space-separated values for the specified type.  Store
        # the values in consecutive elements of the table, with a first index
        # of the type.
        #
        if (sName ~ /^missing_values_(8|16|32|64)bit_(signed|unsigned|float)$/)
        {
            # Extract the size and class of the type from the name.
            #
            match(sName, /^missing_values_(8|16|32|64)bit_(signed|unsigned|float)$/, asParts);

            # Get the type from the Types array.
            #
            sVarType = asTypes[asParts[1],asParts[2]];

            # Get the values by splitting the attribute value string.
            #
            nVars = split(asGroupAttributes["Global",nAtt,"value"], asParts, / +/);

            # Store the values into the table.
            #
            for (nVar = 1 ; nVar <= nVars ; nVar++)
            {
                asMissingValueTable[sVarType,nVar] = asParts[nVar];
            }

            delete asGroupAttributes["Global",nAtt,"name"];
            delete asGroupAttributes["Global",nAtt,"type"];
            delete asGroupAttributes["Global",nAtt,"value"];
        }
    }

    # Examine each "missing_value" attribute in each variable in each group.
    # If it matches the entries for the variable type in the fill value table,
    # delete the attribute.
    #
    for (nGroup = 1 ; nGroup <= nNumGroups ; nGroup++)
    {
        sGroupPath = asGroups[nGroup];

        nVars = anGroupNumVariables[sGroupPath];

        for (nVar = 1 ; nVar <= nVars ; nVar++)
        {
            # Get the number of attributes for the variable.
            #
            nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

            # Get the variable type.  If the variable has an attribute named
            # "_Unsigned" that has a value of "true", make the variable type
            # unsigned.
            #
            sVarType = asGroupVariables[sGroupPath,nVar,"type"];

            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("_Unsigned" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    if ("true" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"])
                    {
                        sVarType = "unsigned " sVarType;
                    }

                    break;
                }
            }

            # If there is a "missing_value" attribute, compare the values
            # against the table.  If there is a match, delete the attribute.
            #
            for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
            {
                if ("missing_value" == asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
                {
                    # Get the value string.  If it is equal to "n/a", move on.
                    #
                    sValue = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

                    if ("n/a" == sValue)
                    {
                        continue;
                    }

                    # Split the value string into parts.  If there is an exact
                    # match with the table, delete the attribute.  Add one to
                    # the number of parts found so that one extra match will be
                    # made, verifying that the end of the table for the type
                    # was found.
                    #
                    nParts = split(sValue, asParts, / +/) + 1;

                    for (nPart = 1 ; nPart <= nParts ; nPart++)
                    {
                        if (asParts[nPart] != asMissingValueTable[sVarType,nPart])
                        {
                            break;
                        }
                    }

                    # All the values matched, so delete the attribute.
                    #
                    if (nPart > nParts)
                    {
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];
                        delete asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];
                    }

                    break;
                }
            }
        }
    }

    # If no missing value table was collected, unset the CollectMissingValues
    # flag.
    #
    sValue = 0;

    for (key in asMissingValueTable)
    {
        sValue = 1;
        
        break;
    }

    if (0 == sValue)
    {
        nCollectMissingValues = 0;

        delete asMissingValueTable;
    }
}


# Write out the preamble section.
#
function generateFilePreamble(asFillValueTable, asMissingValueTable, \
                              asTypes, asValues, nI, nJ, nMeanings, sTmp, sType) # local variables
{
    # Initialize the local variables that need it.
    #
    delete asValues;
    delete asTypes;

    asTypes[1]  = "byte";
    asTypes[2]  = "unsigned byte";
    asTypes[3]  = "short";
    asTypes[4]  = "unsigned short";
    asTypes[5]  = "int";
    asTypes[6]  = "unsigned int";
    asTypes[7]  = "long";
    asTypes[8]  = "unsigned long";
    asTypes[9]  = "float";
    asTypes[10] = "double";

    # Write the spreadsheet version.
    #
    printf "SpreadsheetVersion\t1.0\n\n";

    # If collection of fill or missing values was not requested, return.
    #
    if (1 != nCollectFillValues && 1 != nCollectMissingValues)
    {
        return;
    }

    # Write the settings section, with fill values and missing values
    # subsections as appropriate.
    #
    printf "Settings\n";

    if (1 == nCollectFillValues)
    {
        # Print the subsection line.
        #
        printf "\tFillValues\tType\tValue\n";

        # Print the line for the fill value for each type that is present in
        # the table.
        #
        for (nI = 1 ; nI <= 10 ; nI++)
        {
            sType = asTypes[nI];

            if (sType in asFillValueTable)
            {
                printf "\t\t%s\t%s\n", sType, asFillValueTable[sType];
            }
        }

        printf "\n";
    }

    if (1 == nCollectMissingValues)
    {
        # Start the subsection line.
        #
        printf "\tMissingValues\tType";
        
        # Print the column header meanings.
        #
        nI = 1;

        while (sTmp = asMissingValueTable["meaning",nI])
        {
            printf "\t%s", sTmp;

            nI++;
        }

        # Write the ending newline for the subsection.
        #
        printf "\n";

        # Get the number of meanings.
        #
        nMeanings = nI - 1;

        # Print the values by type.
        #
        for (nI = 1 ; nI <= 10 ; nI++)
        {
            sType = asTypes[nI];

            printf "\t\t%s", sType;

            for (nJ = 1 ; nJ <= nMeanings ; nJ++)
            {
                printf "\t%s", asMissingValueTable[sType,nJ];
            }

            printf "\n";
        }

        printf "\n";
    }
}


# Write out the contents for a group.
#
function generateGroupOutput(sGroupPath, \
                             anVariableIsCoordinate, sGroupName) # local variables.
{
    # Initialize local variables.
    #
    delete anVariableIsCoordinate;

    sGroupName = sGroupPath;

    # Form the name for the group and write the group section line.  If the
    # group is not the "Global" group, strip "Global/" from the start of the
    # group path.
    #
    sub(/^Global\//, "", sGroupName);

    printf "\nGroup\t%s\n", sGroupName;

    # Generate output for the group attributes.
    #
    generateAttributeOutput(sGroupPath);

    # Generate output for the group dimensions.
    #
    generateDimensionOutput(sGroupPath);

    # Determine which variables are coordinates.
    #
    createVariableIsCoordinateArray(sGroupPath, anVariableIsCoordinate);

    # Generate output for the group coordinates.
    #
    generateCoordinateOutput(sGroupPath, anVariableIsCoordinate);

    # Generate output for the group variables.
    #
    generateVariableOutput(sGroupPath, anVariableIsCoordinate);
}


# Write out the attributes for a group.
#
function generateAttributeOutput(sGroupPath, \
                                 nAtts, nI, sName) # local variables
{
    # Write out the attributes subsection line.
    #
    printf "\tAttributes\tName(Att)\tType(Att)\tValue(Att)\n";

    # Write out the information for each attribute in the group.
    #
    nAtts = anGroupNumAttributes[sGroupPath];

    for (nI = 1 ; nI <= nAtts ; nI++)
    {
        sName = asGroupAttributes[sGroupPath,nI,"name"];

        if ("" != sName)
        {
            printf "\t\t%s\t%s\t%s\n", sName, asGroupAttributes[sGroupPath,nI,"type"], asGroupAttributes[sGroupPath,nI,"value"];
        }
    }

    # Write a trailing blank line.
    #
    printf "\n";
}


# Write out the dimensions for a group.
#
function generateDimensionOutput(sGroupPath, \
                                 nDims, nI, sLength) # local variables
{
    # Write out the dimensions subsection line.
    #
    printf "\tDimensions\tName(Dim)\tLength(Dim)\n";

    # Write out the information for each dimension in the group.
    #
    nDims = anGroupNumDimensions[sGroupPath];

    for (nI = 1 ; nI <= nDims ; nI++)
    {
        # Get the length.  If it is "0", change it to "unlimited".
        #
        sLength = asGroupDimensions[sGroupPath,nI,"length"];

        if (sLength == "0")
        {
            sLength = "unlimited";
        }

        printf "\t\t%s\t%s\n", asGroupDimensions[sGroupPath,nI,"name"], sLength;
    }

    # Write a trailing blank line.
    #
    printf "\n";
}


# Write out the coordinates for a group.
#
function generateCoordinateOutput(sGroupPath, anVariableIsCoordinate, \
                                  asColumnHeaders, asParts, nAtts, nColumns, nI, nJ, nVars, sColumnName, sColumnType, \
                                  sDisplayName, sVarType) # local variables
{
    # Get the column headers for the coordinates for the group.
    #
    createCoordinateColumnHeaders(sGroupPath, anVariableIsCoordinate, asColumnHeaders);

    # Get the number of columns.
    #
    nColumns = 0;

    for (nI in asColumnHeaders)
    {
        nColumns++;
    }

    # Write out the coordinates subsection line.
    #
    printf "\tCoordinates\tShape(Coord)\tType(Coord)";

    for (nI = 1 ; nI <= nColumns ; nI++)
    {
        # Get the name, type, and display name for the column.
        #
        split(asColumnHeaders[nI], asParts, /:/);

        sColumnName  = asParts[1];
        sColumnType  = asParts[2];
        sDisplayName = asParts[3];

        # If there is no display name, set it to the name.
        #
        if (!sDisplayName)
        {
            sDisplayName = sColumnName;
        }

        # If the type is not "string" or "type", append ":<type>" to the display
        # name.
        #
        if ("string" != sColumnType && "type" != sColumnType)
        {
            sDisplayName = sDisplayName ":" sColumnType;
        }

        # Add the display name to the output line.
        #
        printf "\t%s", sDisplayName;
    }

    # Finish the subsection line.
    #
    printf "\n";

    # Write out the info for each coordinate variable.
    #
    nVars = anGroupNumVariables[sGroupPath];

    for (nI = 1 ; nI <= nVars ; nI++)
    {
        # If the variable is not a coordinate, skip it.
        #
        if (!anVariableIsCoordinate[nI])
        {
            continue;
        }

        # Get the variable type.  If there is a variable attribute named
        # "_Unsigned", and if its value is "true", make the variable type
        # unsigned.
        #
        sVarType = asGroupVariables[sGroupPath,nI,"type"];

        nAtts = anGroupVariableNumAttributes[sGroupPath,nI];

        for (nJ = 1 ; nJ <= nAtts ; nJ++)
        {
            if ("_Unsigned" == asGroupVariableAttributes[sGroupPath,nI,nJ,"name"])
            {
                if ("true" == asGroupVariableAttributes[sGroupPath,nI,nJ,"value"])
                {
                    sVarType = "unsigned " sVarType;
                }
            }

            break;
        }

        # Write the initial part of the entry for the coordinate.
        #
        printf "\t\t%s\t%s", asGroupVariables[sGroupPath,nI,"shape"], sVarType;

        # Write out the other columns.
        #
        outputVariableColumns(sGroupPath, nI, asColumnHeaders);
    }

    # Write out a final empty line.
    #
    printf "\n";
}


# Write out the variables for a group.
#
function generateVariableOutput(sGroupPath, anVariableIsCoordinate, \
                                asColumnHeaders, asParts, nAtts, nColumns, nI, nJ, nVars, sColumnName, sColumnType, \
                                sDisplayName, sVarType) # local variables
{
    # Get the column headers for the variables for the group.
    #
    createVariableColumnHeaders(sGroupPath, anVariableIsCoordinate, asColumnHeaders);

    # Get the number of columns.
    #
    nColumns = 0;

    for (nI in asColumnHeaders)
    {
        nColumns++;
    }

    # Write out the variables subsection line.
    #
    printf "\tVariables\tName(Var)\tType(Var)\tShape(Var)";

    for (nI = 1 ; nI <= nColumns ; nI++)
    {
        # Get the name, type, and display name for the column.
        #
        split(asColumnHeaders[nI], asParts, /:/);

        sColumnName  = asParts[1];
        sColumnType  = asParts[2];
        sDisplayName = asParts[3];

        # If there is no display name, set it to the name.
        #
        if (!sDisplayName)
        {
            sDisplayName = sColumnName;
        }

        # If the type is not "string" or "type", append ":<type>" to the display
        # name.
        #
        if ("string" != sColumnType && "type" != sColumnType)
        {
            sDisplayName = sDisplayName ":" sColumnType;
        }

        # Add the display name to the output line.
        #
        printf "\t%s", sDisplayName;
    }

    # Finish the subsection line.
    #
    printf "\n";

    # Write out the info for each variable.
    #
    nVars = anGroupNumVariables[sGroupPath];

    for (nI = 1 ; nI <= nVars ; nI++)
    {
        # If the variable is a coordinate, skip it.
        #
        if (1 == anVariableIsCoordinate[nI])
        {
            continue;
        }

        # Get the variable type.  If there is a variable attribute named
        # "_Unsigned", and if its value is "true", make the variable type
        # unsigned.
        #
        sVarType = asGroupVariables[sGroupPath,nI,"type"];

        nAtts = anGroupVariableNumAttributes[sGroupPath,nI];

        for (nJ = 1 ; nJ <= nAtts ; nJ++)
        {
            if ("_Unsigned" == asGroupVariableAttributes[sGroupPath,nI,nJ,"name"])
            {
                if ("true" == asGroupVariableAttributes[sGroupPath,nI,nJ,"value"])
                {
                    sVarType = "unsigned " sVarType;
                }

                break;
            }
        }

        # Write the initial part of the entry for the variable.
        #
        printf "\t\t%s\t%s\t%s", asGroupVariables[sGroupPath,nI,"name"], sVarType, asGroupVariables[sGroupPath,nI,"shape"];
 
        # Write out the other columns.
        #
        outputVariableColumns(sGroupPath, nI, asColumnHeaders);
    }

    # Write out a final empty line.
    #
    printf "\n";
}


# Create an array that identifies the variables that are coordinates.
#
function createVariableIsCoordinateArray(sGroupPath, anVariableIsCoordinate, \
                                         nVar, nNumVars) # local variables
{
    # Clear the argument array.
    #
    delete anVariableIsCoordinate;

    # For each variable in the group, find out if the name and the shape are
    # identical.  If the answer is yes, put an entry in the argument
    # anVariableIsCoordinate array.
    #
    nNumVars = anGroupNumVariables[sGroupPath];

    for (nVar = 1 ; nVar <= nNumVars ; nVar++)
    {
        if (asGroupVariables[sGroupPath,nVar,"name"] == asGroupVariables[sGroupPath,nVar,"shape"])
        {
            anVariableIsCoordinate[nVar] = 1;
        }
    }
}


# Convert a name from lower_bar to CamelCase.  Returns the converted name.
#
function toCamelCase(sName, \
                     asSubs, nIndex, nSubs, sChar) # local variables
{
    # If the name starts with an underscore, return it un-modified.
    #
    if (sName ~ /^_/)
    {
        return sName;
    }

    # If the name has uppercase characters in it already, return it unmodified.
    #
    if (sName ~ /[[:upper:]]/)
    {
        return sName;
    }

    # Break the name apart on '_' characters.
    #
    nSubs = split(sName, asSubs, /_/);

    # Convert the first character in each substring to upper case.
    #
    for (nIndex = 1 ; nIndex <= nSubs ; nIndex++)
    {
        sChar = substr(asSubs[nIndex], 1, 1);
        sChar = toupper(sChar);

        sub(/^./, sChar, asSubs[nIndex]);
    }

    # Reform the name from the parts.
    #
    sName = "";

    for (nIndex = 1 ; nIndex <= nSubs ; nIndex++)
    {
        sName = sName asSubs[nIndex];
    }

    # Return the camel-cased name.
    #
    return sName;
}


# Determine the column headers needed for the variables in the group.  The
# entries in the column headers array are of the form
# <name>:<type>[:<display name>].
# In the case of attributes that match the type of the corresponding variable,
# the type name will be "type".  The display name is optional.
#
function createVariableColumnHeaders(sGroupPath, anVariableIsCoordinate, asColumnHeaders, \
                                     asAttributeInfo, nAtt, nNumAtts, nVar, nNumVars, sAttType, sVarType, sAttName, \
                                     sDisplayName) # local variables
{
    # Initialize the local variables.
    #
    delete asAttributeInfo;

    # Clear the argument ColumnHeaders array.
    #
    delete asColumnHeaders;

    # Step through the attributes for each non-coordinate variable, and collect
    # the full set of attribute names and types.
    #
    nNumVars = anGroupNumVariables[sGroupPath];

    for (nVar = 1 ; nVar <= nNumVars ; nVar++)
    {
        # If the variable is a coordinate, skip it.
        #
        if (1 == anVariableIsCoordinate[nVar])
        {
            continue;
        }

        # Get the variable type.
        #
        sVarType = asGroupVariables[sGroupPath,nVar,"type"];

        # Store the type of each attribute in the AttributeInfo array indexed by
        # attribute name.  If the type of the attribute matches the type of the
        # variable, use the word "type" instead of the actual type.  If the type
        # is empty, skip it.
        #
        nNumAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

        for (nAtt = 1 ; nAtt <= nNumAtts ; nAtt++)
        {
            sAttType = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];

            if (!sAttType)
            {
                continue;
            }

            if (sAttType == sVarType)
            {
                sAttType = "type";
            }

            asAttributeInfo[asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"]] = sAttType;
        }
    }

    # Use a preferred order for column headers for "units", "valid_min",
    # "valid_max", "_FillValue", "long_name", and "source", if they are present.
    #
    nAtt = 1;

    if ("units" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "units:string:Units";

        delete asAttributeInfo["units"];
    }

    if ("valid_min" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "valid_min:type:ValidMin";

        delete asAttributeInfo["valid_min"];
    }

    if ("valid_max" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "valid_max:type:ValidMax";

        delete asAttributeInfo["valid_max"];
    }

    if ("_FillValue" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "_FillValue:type:FillValue";

        delete asAttributeInfo["_FillValue"];
    }

    if ("long_name" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "long_name:string:LongName";

        delete asAttributeInfo["long_name"];
    }

    if ("source" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "source:string:Source";

        delete asAttributeInfo["source"];
    }

    # If there is an entry for "_Unsigned", remove it.  This is handled
    # differently.
    #
    if ("_Unsigned" in asAttributeInfo)
    {
        delete asAttributeInfo["_Unsigned"];
    }

    # Store any remaining elements of the AttributeInfo array as column headers
    # after the ones already entered.
    #
    for (sAttName in asAttributeInfo)
    {
        # Create a camel-cased display name.
        #
        sDisplayName = toCamelCase(sAttName);

        asColumnHeaders[nAtt++] = sAttName ":" asAttributeInfo[sAttName] ":" sDisplayName;
    }
}


# Determine the column headers needed for the coordinates in the group.  The
# entries in the column headers array are of the form
# <name>:<type>[:<display name>].
# In the case of attributes that match the type of the corresponding variable,
# the type name will be "type".  The display name is optional.
#
function createCoordinateColumnHeaders(sGroupPath, anVariableIsCoordinate, asColumnHeaders, \
                                       asAttributeInfo, nAtt, nNumAtts, nVar, nNumVars, sAttType, sVarType, sAttName,
                                       sDisplayName) # local variables
{
    # Initialize the local variables.
    #
    delete asAttributeInfo;

    # Clear the argument ColumnHeaders array.
    #
    delete asColumnHeaders;

    # Step through the attributes for each non-coordinate variable, and collect
    # the full set of attribute names and types.
    #
    nNumVars = anGroupNumVariables[sGroupPath];

    for (nVar = 1 ; nVar <= nNumVars ; nVar++)
    {
        # If the variable is not a coordinate, skip it.
        #
        if (!anVariableIsCoordinate[nVar])
        {
            continue;
        }

        # Get the variable type.
        #
        sVarType = asGroupVariables[sGroupPath,nVar,"type"];

        # Store the type of each attribute in the AttributeInfo array indexed by
        # attribute name.  If the type of the attribute matches the type of the
        # variable, use the word "type" instead of the actual type.  If the type
        # is empty, skip it.
        #
        nNumAtts = anGroupVariableNumAttributes[sGroupPath,nVar];

        for (nAtt = 1 ; nAtt <= nNumAtts ; nAtt++)
        {
            sAttType = asGroupVariableAttributes[sGroupPath,nVar,nAtt,"type"];

            if (!sAttType)
            {
                continue;
            }

            if (sAttType == sVarType)
            {
                sAttType = "type";
            }

            asAttributeInfo[asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"]] = sAttType;
        }
    }

    # Use a preferred order for column headers for "units", "axis", "values",
    # "valid_min", "valid_max", and "long_name" if they are present.
    #
    nAtt = 1;

    if ("units" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "units:string:Units";

        delete asAttributeInfo["units"];
    }

    if ("axis" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "axis:string:Axis";

        delete asAttributeInfo["axis"];
    }

    if ("values" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "values:type:Values";

        delete asAttributeInfo["values"];
    }

    if ("valid_min" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "valid_min:type:ValidMin";

        delete asAttributeInfo["valid_min"];
    }

    if ("valid_max" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "valid_max:type:ValidMax";

        delete asAttributeInfo["valid_max"];
    }

    if ("long_name" in asAttributeInfo)
    {
        asColumnHeaders[nAtt++] = "long_name:string:LongName";

        delete asAttributeInfo["long_name"];
    }

    # If the attribute name "_Unsigned" appears in the list, remove it.  It is
    # handled differently.
    #
    if ("_Unsigned" in asAttributeInfo)
    {
        delete asAttributeInfo["_Unsigned"];
    }

    # Store any remaining elements of the AttributeInfo array as column headers
    # after the ones already entered.
    #
    for (sAttName in asAttributeInfo)
    {
        # Create a camel-cased display name.
        #
        sDisplayName = toCamelCase(sAttName);

        asColumnHeaders[nAtt++] = sAttName ":" asAttributeInfo[sAttName] ":" sDisplayName;
    }
}


# Write out attribute values for a variable based on the column headers present.
#
function outputVariableColumns(sGroupPath, nVar, asColumnHeaders, \
                               anIndexOfAttribute, sName, nAtt, nAtts, nCol, nCols) # local variables
{
    # Initialize the local variables.
    #
    delete anIndexOfAttribute;

    # Get the number of column headers and the number of attributes for the
    # variable.
    #
    nAtts = anGroupVariableNumAttributes[sGroupPath,nVar];
    nCols = 0;

    for (nCol in asColumnHeaders)
    {
        nCols++;
    }

    # Build an array of the index of each valid attribute in the
    # asGroupVariableAttributes array indexed by attribute name.
    #
    for (nAtt = 1 ; nAtt <= nAtts ; nAtt++)
    {
        if (!asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"])
        {
            continue;
        }

        anIndexOfAttribute[asGroupVariableAttributes[sGroupPath,nVar,nAtt,"name"]] = nAtt;
    }

    # Step through column by column and find a value (if available) to write for
    # the attribute in question.
    #
    for (nCol = 1 ; nCol <= nCols ; nCol++)
    {
        # Write out a tab to delimit the field.
        #
        printf "\t";

        # Get the name for the attribute for the column.
        #
        sName = gensub(/:.*$/, "", "1", asColumnHeaders[nCol]);

        # Find the index for the attribute.
        #
        nAtt = anIndexOfAttribute[sName];

        # If the index is valid, get the value and write it out and move on.
        #
        if (0 < nAtt)
        {
            printf "%s", asGroupVariableAttributes[sGroupPath,nVar,nAtt,"value"];

            continue;
        }
    }

    # Write a newline to finish the line.
    #
    printf "\n";
}


# Convert XML "escape" sequences for '&', '<', and '>' to their target
# characters.
#
function convertFromXmlCharacters(sString)
{
    # Convert any XML '&gt;' sequences into '>' characters.
    #
    gsub(/&gt;/, ">", sString);

    # Convert any XML '&lt;' sequences into '<' characters.
    #
    gsub(/&lt;/, "<", sString);

    # As a last step, convert any XML '&amp;' sequences into '&' characters.
    # This is done last to prevent unintended embedded conversions.
    #
    gsub(/&amp;/, "\\&", sString);

    # Return the result.
    #
    return sString;
}
