# This file contains awk patterns and actions used to convert a
# tab-separated-values (TSV) file that follows a specific template into a
# netCDF Markup Language (NcML) file.

# Strip leading and trailing whitespace from the string.
#
function strip(sString)
{
    sub(/^[[:space:]]+/, "", sString);
    sub(/[[:space:]]+$/, "", sString);

    return sString;
}


# This BEGIN block initializes an array of characters indexed by their XML
# escape sequences, and another array of XML escape sequences indexed by their
# characters.  These arrays are used by the convertToXmlCharacters function.
#
BEGIN \
{
    asCharByXmlSeq["&amp;"]  = "&";
    asCharByXmlSeq["&apos;"] = "'";
    asCharByXmlSeq["&gt;"]   = ">";
    asCharByXmlSeq["&lt;"]   = "<";
    asCharByXmlSeq["&quot;"] = "\"";

    asXmlSeqByChar["&"]  = "&amp;";
    asXmlSeqByChar["'"]  = "&apos;";
    asXmlSeqByChar[">"]  = "&gt;";
    asXmlSeqByChar["<"]  = "&lt;";
    asXmlSeqByChar["\""] = "&quot;";
}


# Convert characters in the argument string that need to be converted to XML
# escape sequences and return the result.  Warn the user about the presence of
# any XML escape sequences that won't be allowed by NcML-to-netCDF software.
#
function convertToXmlCharacters(sString, \
                                asParts, asParts2, nCharNum, nIsOK, sOutput, sTmp, sXmlSeq) # local variables
{
    # Scan through the argument string, locating any XML escape sequences that
    # already exist.  If any of them specify characters outside the ASCII set,
    # warn the user.
    #
    sOutput = sString;

    do
    {
        # Split the string into the part preceding an ampersand, and the part
        # that starts with the ampersand.
        #
        match(sOutput, /^([^&]+)?(&.+)?$/, asParts);

        # If there is an ampersand, see if it is the start of an XML escape
        # sequence.  If it is and the character it represents is not allowed,
        # warn the user.  Keep scanning.
        #
        if (2 in asParts)
        {
            # Split the string into a part that could be an XML escape sequence,
            # and the part that follows it.
            #
            match(asParts[2], /^(&#[Xx][[:xdigit:]]+;|&#[[:digit:]]+;|&[[:alnum:]]+;)?(.+)?$/, asParts2);

            # If there is a possible escape sequence, set the XmlSeq variable
            # to the escape sequence found and examine it.  Otherwise, set the
            # XmlSeq variable to the string "&" so that the later code will
            # move past it.
            #
            sXmlSeq = "&";

            if (1 in asParts2)
            {
                nIsOK   = 1;
                sXmlSeq = asParts2[1];

                # If the XML escape sequence is a number, get the number.  If
                # it is outside the valid range (9,10,13,32-126), set IsOK to
                # 0.  Otherwise, see if the sequence matches one of the allowed
                # escape sequences.  If not, set IsOK to 0.
                #
                if (sXmlSeq ~ /^&#/)
                {
                    if (sXmlSeq ~ /^&#[xX]/) # A hex number
                    {
                        # Get the number part.  Add a leading zero.
                        #
                        sTmp = gensub(/^&#([^;]+);$/, "0\\1", "1", sXmlSeq);
                    }
                    else # A base-10 number
                    {
                        # Get the number part.  Strip off any leading zeros.
                        #
                        sTmp = gensub(/^&#0*([^;]+);$/, "\\1", "1", sXmlSeq);
                    }

                    # Get the value of the number string.  If it is outside the
                    # valid range, set IsOK to 0.
                    #
                    nCharNum = strtonum(sTmp);

                    do
                    {
                        if (9  == nCharNum)                  break;
                        if (10 == nCharNum)                  break;
                        if (13 == nCharNum)                  break;
                        if (31 < nCharNum && 127 > nCharNum) break;

                        nIsOK = 0;
                    }
                    while (0);
                }
                else if (!(sXmlSeq in asCharByXmlSeq))
                {
                    nIsOK = 0;
                }

                # If the sequence is not OK, print a warning.
                #
                if (0 == nIsOK)
                {
                    printf "Warning: XML escape sequence %s in line %d is not allowed.\n", sXmlSeq, NR > "/dev/stderr";
                }
            }

            # Set the input string to the following part after the XML sequence
            # has been removed.  Continue to scan.
            #
            sOutput = substr(sOutput, 1 + length(sXmlSeq));

            continue;
        }

        # If execution reaches this point, the entire string has been scanned.
        # Break out of the loop.
        #
        break;
    }
    while (1);


    # Scan through the argument string, locating any '&', "'", '"', '<', or '>'
    # characters.  Convert them to XML escape sequences.  Don't convert any '&'
    # characters that are already part of XML escape sequences.
    #
    sOutput = "";

    do
    {
        # Split the string into the part preceding , and the part that starts
        # with the one of the characters to be converted.
        #
        match(sString, /^([^&'"<>]+)?([&'"<>].*)?$/, asParts);

        # If there is a preceding part, append it to the output string.
        #
        if (1 in asParts)
        {
            sOutput = sOutput asParts[1];
        }

        # If a character was found, process it and keep scanning.
        #
        if (2 in asParts)
        {
            sTmp = asParts[2];

            # Split the string into a part that is an XML escape sequence,
            # and the part that follows it.
            #
            match(sTmp, /^(&#[Xx][[:xdigit:]]+;|&#[[:digit:]]+;|&[[:alnum:]]+;)?(.+)?$/, asParts2);

            # If the character found is an '&' that is the start of an XML
            # escape sequence, append the escape sequence to the output string
            # and set Tmp to the remaining characters.  Otherwise, append the
            # proper XML escape sequence to the output string and set Tmp to
            # the remaining characters.
            #
            if (1 in asParts2)
            {
                sOutput = sOutput asParts2[1];
                sTmp    = asParts2[2];
            }
            else
            {
                sOutput = sOutput asXmlSeqByChar[substr(sTmp, 1, 1)];
                sTmp    = substr(sTmp, 2);
            }

            # Set the input string to the following part after the character or
            # XML sequence has been removed.  Continue to scan.
            #
            sString = sTmp;

            continue;
        }

        # If execution reaches this point, the entire string has been scanned.
        # Break out of the loop.
        #
        break;
    }
    while (1);

    # Return the converted string.
    #
    return sOutput;
}

# Change CamelCase to lower_bar.  If the string starts with a "_", then do
# nothing.
#
function toLowerBar(sString)
{
    # If the first character is "_", this is a special name.  Return it without
    # modification.
    #
    if (sString ~ /^_/)
    {
        return sString;
    }

    # Prefix each upper-case character with a '_' character.
    #
    gsub(/([[:upper:]])/, "_&", sString);

    # Change all characters to lower-case.
    #
    sString = tolower(sString);

    # Strip any leading '_' character.
    #
    sub(/^_/, "", sString);

    return sString;
}


# Convert each of the names in the array to lower_bar format.  Handle the
# special names "_FillValue", "Name(.*)", "Shape(.*)", "Type(.*)", and
# "Length(.*)".
#
function convertNamesToLowerBar(asNames, \
                                sName, key) # local variables
{
    # Loop through every element of the Names array and convert the names to
    # lower_bar format.  If the name is "FillValue", change it to "_FillValue".
    #
    for (key in asNames)
    {
        sName = asNames[key];

        if ("FillValue" == sName)
        {
            asNames[key] = "_FillValue";
        }
        else if (sName ~ /^([Nn]ame|[Ss]hape|[Tt]ype|[Ll]ength|[Vv]alue)\([[:alpha:]]+\)$/)
        {
            sub(/\([[:alpha:]]+\)$/, "", sName);

            asNames[key] = "@" toLowerBar(sName);
        }
        else
        {
            asNames[key] = toLowerBar(sName);
        }
    }
}


# Verify that the argument string contains only letters, digits and underscores,
# and that the first character (possibly prefixed by an underscore) is a letter.
# Return 1 if true, and 0 if false.
#
function verifyName(sName)
{
    return (sName ~ /^_?[[:alpha:]]\w*$/);
}


# Form an array that contains the input fields, indexed by field number.  Start
# at the field number specified, if the argument is provided.  Don't include
# empty fields.
#
function buildInputArray(asArray, nStartField)
{
    # Clear the array to get rid of any old contents.
    #
    delete asArray;

    # If the start field is not specified (equals zero), set the StartField
    # value to 1.
    #
    if (0 == nStartField)
    {
        nStartField = 1;
    }

    # Store the fields into the array, stripping off any leading or trailing
    # whitespace.
    #
    for (; nStartField <= NF ; nStartField++)
    {
        if ("" == $nStartField) continue;

        asArray[nStartField] = strip($nStartField);
    }
}


# Remove empty elements from the input array.
#
function removeEmptyElements(asArray, \
                             key, anEmpty) # local variables
{
    # Clear the local variables.
    #
    key = "";
    delete anEmpty;

    # Build an array that flags the empty elements in Array.
    #
    for (key in asArray)
    {
        if ("" == asArray[key])
        {
            anEmpty[key] = 1;
        }
    }

    # Delete the elements of Array that are empty.
    #
    for (key in anEmpty)
    {
        delete asArray[key];
    }
}


# Build an array from an input array that is indexed by field number that
# contains field numbers indexed by contents.  Don't include empty or blank
# elements.  Note that if the contents are not unique, the last instance
# processed will be the only field number that will be stored.  (Processing
# order is not sequential!)
#
function buildReverseMap(asInput, anOutput, \
                            nField, sContent) # local variables
{
    # Clear the local variables.
    #
    nField = 0;
    sContent = "";

    # Clear the output array.
    #
    delete anOutput;

    # Step through each element of the input array.
    #
    for (nField in asInput)
    {
        # Get the current element contents.
        #
        sContent = asInput[nField];

        # Strip leading and trailing whitespace.
        #
        sContent = strip(sContent);

        # If the string is empty, skip it.
        #
        if ("" == sContent) continue;

        # Store the field number into the Output array indexed by the contents.
        #
        anOutput[sContent] = nField;
    }
}


# Validate the array of field numbers indexed by name against the array of
# defined elements (which are also indexed by name).  If the AllowMissing
# argument is set to "true", only defined headers that are prefixed with an '@'
# character are required.  If the AllowExtra argument is set to "true",
# user-defined header names are allowed.
#
function checkHeaderNames(anFieldsByName, asElementsByName, sAllowMissing, sAllowExtra, \
                          asSplit, nFields, nFound, nIsRequired, nMissing, nRequired, nRetVal, sName, sType, field, key) # local variables
{
    # Clear the local variables.
    #
    delete asSplit;

    nFields     = 0;
    nFound      = 0;
    nIsRequired = 0;
    nMissing    = 0;
    nRequired   = 0;
    nRetVal     = 0;
    sType       = "";
    field       = "";
    key         = "";

    # Examine each key in the ElementsByName array.
    #
    for (key in asElementsByName)
    {
        # If the element referenced by key begins with an '@' character, this
        # header is required.  Set the IsRequired flag, and increment the
        # required count.
        #
        if (asElementsByName[key] ~ /^@/)
        {
            nIsRequired = 1;
            nRequired++;
        }
        else
        {
            nIsRequired = 0;
        }

        # If a field exists that has the key name, increment the Found count.
        # Decrement the required count by the value of the IsRequired flag.
        # If the header is not required, this won't change the required count.
        # If no field exists that has the key name, increment the missing count.
        #
        field = (key in anFieldsByName) ? anFieldsByName[key] : "";

        if ("" != field)
        {
            nRequired -= nIsRequired;

            nFound++;
        }
        else
        {
            nMissing++;
        }
    }

    # If required headers weren't found, report the error and return 0.
    #
    if (0 != nRequired)
    {
        printf "Error: Required column headers are missing at line %d\n", NR > "/dev/stderr";
        return 0;
    }

    # If there were missing optional headers, but that is not allowed, report
    # the error and return 0.
    #
    if ("true" != sAllowMissing && 0 < nMissing)
    {
        printf "Error: All standard column headers are required, but one or more are missing at line %d\n", NR > "/dev/stderr";
        return 0;
    }

    # Count the number of fields.
    # (The length function won't work here for some reason!)
    #
    for (key in anFieldsByName)
    {
        nFields++;
    }

    # If there were extra headers, but that is not allowed, report the error and
    # return 0.
    #
    if ("true" != sAllowExtra && nFound < nFields)
    {
        printf "Error: User-defined column headers are not allowed, but one or more are present at line %d\n", NR > "/dev/stderr";
        return 0;
    }

    # Validate the names for the fields.  They must all have the form
    # <name>[:<type>], where the name must be composed entirely of '_', letters,
    # and numbers, and cannot begin with a number.  The type, if present, must
    # be the word "type" or a valid type name.  If a name fails validation,
    # report it and return 0.
    #
    nRetVal = 1;

    for (key in anFieldsByName)
    {
        # Split the field contents into parts using ":" as the delimiter.
        #
        nFound = split(key, asSplit, ":");
        
        # If there was more than one ":", report the failure and move to the
        # next field.
        #
        if (2 < nFound)
        {
            printf "Error: Column header \"%s\" is malformed at line %d\n", key, NR > "/dev/stderr";

            nRetVal = 0;

            continue;
        }

        # The first part is the name part.  If it is not a reserved name that
        # is preceded by an "@", verify the name.  If the name part is not a
        # netCDF conforming name, report the failure and move to the next
        # field.
        #
        sName = asSplit[1];
        
        if (sName !~ /^@(name|shape|type|value|length)$/)
        {
            if (0 == verifyName(sName))
            {
                printf "Error: Column header \"%s\" specifies an invalid name at line %d\n", key, NR > "/dev/stderr";

                nRetVal = 0;

                continue;
            }
        }

        # If the name has a type spec, validate it.  If it fails, report the
        # failure and move to the next field.
        #
        if (2 == nFound)
        {
            if (asSplit[2] !~ /^(type|string|float|double|((unsigned )?(byte|short|int|long)))$/)
            {
                printf "Error: Column header \"%s\" specifies an invalid type at line %d\n", key, NR > "/dev/stderr";

                nRetVal = 0;

                continue;
            }
        }
    }

    # Check for a "scaled_type" field.  If there is one, set the first bit of
    # the Found flag.
    #
    nFound = ("scaled_type" in anFieldsByName) ? 1 : 0;

    # Check for a "add_offset" field.  If there is one, set the second bit of
    # the Found flag.
    #
    nFound += ("add_offset" in anFieldsByName) ? 2 : 0;

    # Check for a "scale_factor" field.  If there is one, set the third bit of
    # the Found flag.
    #
    nFound += ("scale_factor" in anFieldsByName) ? 4 : 0;

    # If either "add_offset" or "scale_factor" are present but the scaled_type
    # field isn't, report a failure.
    #
    if (2 == nFound || 6 == nFound)
    {
        printf "Error: Column header \"add_offset\" has no corresponding \"scaled_type\" column header at line %d\n", NR > "/dev/stderr";

        nRetVal = 0;
    }
    else if (4 == nFound || 6 == nFound)
    {
        printf "Error: Column header \"scale_factor\" has no corresponding \"scaled_type\" column header at line %d\n", NR > "/dev/stderr";

        nRetVal = 0;
    }

    # Return the result value.
    #
    return nRetVal;
}


# Build an array of elements indexed by field number.  If there are user-defined
# elements, handle them.
#
function buildElementsByField(anFieldsByName, asElementsByName, asElementsByField, \
                              key, sElement) # local variables
{
    # Clear the local variables.
    #
    key      = "";
    sElement = "";

    # Clear any old contents from the ElementsByField array.
    #
    delete asElementsByField;

    # Handle each field in the FieldsByName array.
    #
    for (key in anFieldsByName)
    {
        # If there is not a defined element for this name, create a user-defined
        # element.  Otherwise, use the defined element.
        #
        sElement = (key in asElementsByName) ? asElementsByName[key] : "";

        if ("" == sElement)
        {
            # Take the name and de-camelcase it.
            #
            sElement = toLowerBar(key);

            # If there is no ":" character, add a default type spec of "string".
            #
            if (sElement !~ /:/)
            {
                sElement = sElement ":string";
            }
        }

        # Store the element into the ElementsByField array, indexed by the field
        # number.
        #
        asElementsByField[anFieldsByName[key]] = sElement;
    }
}


# Write an attribute element to the current group contents.
#
function addAttribute(sElementInfo, sVariableType, sValue, \
                      sName, sType, asSplit) # local variables
{
    # Clear the local variables.
    #
    delete asSplit;

    sName = "";
    sType = "";

    # Split the element info into the name and type.
    #
    split(sElementInfo, asSplit, ":");

    sName = asSplit[1];
    sType = asSplit[2];

    # If the name is prefixed by an "@", remove it.
    #
    if (sName ~ /^@/)
    {
        sName = substr(sName, 2);
    }

    # If the type is "type", replace it with the argument variable type.
    #
    if ("type" == sType)
    {
        sType = sVariableType;
    }

    # If the type is empty, make it "string".
    #
    if ("" == sType)
    {
        sType = "string";
    }

    # If the type is "string", convert any characters that need converting into
    # XML escape sequences.
    #
    if ("string" == sType)
    {
        sValue = convertToXmlCharacters(sValue);
    }

    # Write the XML attribute entry to the group contents.
    #
    sGroupContents = sprintf("%s%s<attribute name=\"%s\" type=\"%s\" value=\"%s\" />\n",
                             sGroupContents, sIndent, sName, sType, sValue);
}


# Attempt to locate the dimension name passed in within the dimension
# dictionary along the path to the current group.  Return 1 if found, and 0 if
# not.
#
function findDimensionOnPath(sDimension, \
                             nFound, sPath, sDimPath, key) # local variables
{
    # Initialize local variables.
    #
    nFound   = 1;
    sPath    = sGroupPath;
    sDimPath = "";
    key      = 0;

    # While the path is not empty, form a dimension path composed of the group
    # path concatenated with the dimension name separated by a '/' character.
    # Check for the presence of that "dimension path" in the dictionary.  If it
    # is found, return 1.  If not, remove the last element of the group path and
    # try again.
    #
    while (1 == nFound)
    {
        sDimPath = sPath "/" sDimension;

        if (sDimPath in anDimensionDictionary)
        {
            return 1;
        }

        nFound = sub(/\/\w+$/, "", sPath);
    }

    # The dimension wasn't found.  Return 0.
    #
    return 0;
}


# Verify that the shape string passed in is composed of valid dimension names
# that are accessible to the current group.  Return 1 if true and 0 if false.
#
function isShapeValid(sShape, \
                      asDimensions, nRetVal, key) # local variables
{
    # Initialize the local variables.
    #
    delete asDimensions;

    nRetVal = 1; # indicates success.
    key     = 0;

    # Break the shape string up into the individual dimensions.  Any amount of
    # whitespace constitutes a delimiter.
    #
    split(sShape, asDimensions, / +/);

    # Attempt to find each dimension.  Report an error for each dimension that
    # could not be found, and set the return value to 0.
    #
    for (key in asDimensions)
    {
        if (0 == findDimensionOnPath(asDimensions[key]))
        {
            printf "Error: Shape \"%s\" contains undefined dimension \"%s\" at line %d.\n", sShape, asDimensions[key], NR > "/dev/stderr";

            nRetVal = 0;
        }
    }

    return nRetVal;
}


# Construct a coordinates attribute that is a union of a shape and a
# coordinates attribute specification.  Return the new coordinates attribute
# value.
#
function buildCoordinatesAttribute(anCoordinateNames, sShape, sCoordinates, \
                                   anCoordinates, asCoordinates, asParts, nCoordinates, nParts, sName, key) # local variables
{
    # If the shape is empty or zero, there is nothing to do.  Return an empty
    # string.
    #
    if (!sShape)
    {
        # If the coordinates attribute value is not empty, write a warning.
        #
        if ("" != sCoordinates)
        {
            printf "Warning: A coordinates attribute was specified for a scalar variable in line %d.  Discarding.\n", NR > "/dev/stderr";
        }

        return "";
    }

    # Split the shape into parts, and build arrays that map field numbers to
    # names and vice versa for the shape elements that are coordinates.
    #
    nParts = split(sShape, asParts, / +/);

    nCoordinates = 0;

    for (key = 1 ; key <= nParts ; key++)
    {
        sName = asParts[key];

        if (sName in anCoordinateNames)
        {
            anCoordinates[sName]        = ++nCoordinates;
            asCoordinates[nCoordinates] = sName;
        }
    }

    # If the coordinates attribute value is not empty, process it.
    #
    if ("" != sCoordinates)
    {
        # Split the coordinates attribute value into parts.
        #
        nParts = split(sCoordinates, asParts, / +/);

        # If a coordinate name is not found in the list, add it.
        #
        for (key in asParts)
        {
            sName = asParts[key];

            if (!(sName in anCoordinates))
            {
                anCoordinates[sName]        = ++nCoordinates;
                asCoordinates[nCoordinates] = sName;
            }
        }
    }

    # Build a string that contains the coordinate names.
    #
    sCoordinates = asCoordinates[1];

    for (key = 2 ; key <= nCoordinates ; key++)
    {
        sCoordinates = sCoordinates " " asCoordinates[key];
    }

    # Return the merged coordinates attribute value.
    #
    return sCoordinates;
}


# Create and return an XML values element string using the contents of the
# argument values string.  Handle both set of space-separated values and a
# "start=<n> increment=<m>" pair.  If there is an error, warn the user and
# return an empty string.
#
function buildValuesElement(sValuesString, \
                            asParts, sXmlString) # local variables
{
    # Initialize any local variables that need it.
    #
    delete asParts;

    sXmlString = "";

    # If the values string is a start/increment pair, generate the appropriate
    # XML string.  Otherwise, assume the values string is a list of values and
    # generate the XML string appropriate for that case.
    #
    if (sValuesString ~ /^start=[^ ]+ increment=[^ ]+$/)
    {
        # Get the start value and increment value from the values string.
        #
        match(sValuesString,
              /^start=(-*[[:digit:]]+(.[[:digit:]]*)*([eE][+-]*[[:digit:]]+)*) increment=(-*[[:digit:]]+(.[[:digit:]]*)*([eE][+-]*[[:digit:]]+)*)$/,
              asParts);

        # If either the start or increment is not a valid number, print a
        # warning and quit processing.
        #
        if (!asParts[1] || !asParts[4])
        {
            printf "Warning: Invalid start/increment specification for the values attribute in line %d.  Discarding.\n", NR > "/dev/stderr";

            return "";
        }

        # Form the XML string.
        #
        sXmlString = "<values start=\"" asParts[1] "\" increment=\"" asParts[4] "\" />";
    }
    else
    {
        sXmlString = "<values separator=\" \">" sValuesString "</values>";
    }

    # Return the string.
    #
    return sXmlString;
}


# Initialize global variables.
#
BEGIN \
{
    nResult = 0;

    # Set the input field separator to the TAB character.
    #
    FS = "\t";

    # Initialize the spreadsheet version to a value indicating it is unknown.
    #
    sSpreadsheetVersion = "0.0";

    # Set the state machine variables.
    #
    sInSection = "none";

    nGroupDepth    = 0;
    sGroupContents = "";
    sGroupPath     = "";
    sGroupName     = "";

    sIndentSource = "                                                                ";
    sIndent       = "    ";
    nIndentLevel  = 4;

    # Set the maximum and minimum values for each data type.
    #
    asValidMaxByType["byte"]           = "127";
    asValidMaxByType["unsigned byte"]  = "255";
    asValidMaxByType["short"]          = "32767";
    asValidMaxByType["unsigned short"] = "65535";
    asValidMaxByType["int"]            = "2147483647";
    asValidMaxByType["unsigned int"]   = "4294967295";
    asValidMaxByType["long"]           = "9223372036854775807";
    asValidMaxByType["unsigned long"]  = "18446744073709551615";
    asValidMaxByType["float"]          = "3.40282347e+38";
    asValidMaxByType["double"]         = "1.7976931348623157e+308";

    asValidMinByType["byte"]           = "-128";
    asValidMinByType["unsigned byte"]  = "0";
    asValidMinByType["short"]          = "-32768";
    asValidMinByType["unsigned short"] = "0";
    asValidMinByType["int"]            = "-2147483648";
    asValidMinByType["unsigned int"]   = "0";
    asValidMinByType["long"]           = "-9223372036854775808";
    asValidMinByType["unsigned long"]  = "0";
    asValidMinByType["float"]          = "-3.40282347e+38";
    asValidMinByType["double"]         = "-1.7976931348623157e+308";

    # Initialize empty fill value and missing value arrays.
    #
    asFillValuesByType["void"]    = "";
    asMissingValuesByType["void"] = "";
    asMissingValueMeanings[0] = "dummy";

    # Initialize arrays used to handle the input fields.
    #
    asInputArray[0]        = "dummy";
    anFieldByName["dummy"] = 0;
    asNameByField[0]       = "dummy";

    # Initialize the element by name and element by field arrays for each
    # section.  The names that start with "@" are required fields.
    #
    asFillValueElementsByName["type"]  = "@type:string";
    asFillValueElementsByName["value"] = "@value:type";

    asFillValueElementsByField[0]       = "dummy";
    anFillValueFieldsByElement["dummy"] = 0;

    asMissingValueElementsByName["type"] = "@type:string";

    asMissingValueElementsByField[0]       = "dummy";
    anMissingValueFieldsByElement["dummy"] = 0;

    asAttributeElementsByName["@name"]  = "@name:string";
    asAttributeElementsByName["@type"]  = "@type:string";
    asAttributeElementsByName["@value"] = "@value:type";

    asAttributeElementsByField[0]       = "dummy";
    anAttributeFieldsByElement["dummy"] = 0;

    asDimensionElementsByName["@name"]   = "@name:string";
    asDimensionElementsByName["@length"] = "@length:string";

    asDimensionElementsByField[0]       = "dummy";
    anDimensionFieldsByElement["dummy"] = 0;

    anDimensionDictionary["dummy"] = 0;

    # Coordinate elements are a bit different, since the name and shape must
    # match.  Both are marked as required (with the '@' prefix), but program
    # logic enforces the requirement that only one or the other must be
    # present, and that if both are present, that they must match.
    #
    asCoordinateElementsByName["@name"]         = "@name:string";
    asCoordinateElementsByName["@shape"]        = "@shape:string";
    asCoordinateElementsByName["@type"]         = "@type:string";
    asCoordinateElementsByName["units"]         = "units:string";
    asCoordinateElementsByName["valid_min"]     = "valid_min:string";
    asCoordinateElementsByName["valid_max"]     = "valid_max:string";
    asCoordinateElementsByName["axis"]          = "axis:string";
    asCoordinateElementsByName["values"]        = "values:type";
    asCoordinateElementsByName["long_name"]     = "long_name:string";
    asCoordinateElementsByName["bounds"]        = "bounds:string";
    asCoordinateElementsByName["calendar"]      = "calendar:string";
    asCoordinateElementsByName["climatology"]   = "climatology:string";
    asCoordinateElementsByName["compress"]      = "compress:string";
    asCoordinateElementsByName["formula_terms"] = "formula_terms:string";
    asCoordinateElementsByName["leap_month"]    = "leap_month:int";
    asCoordinateElementsByName["leap_year"]     = "leap_year:int";
    asCoordinateElementsByName["month_lengths"] = "month_lengths:int";
    asCoordinateElementsByName["positive"]      = "positive:string";
    asCoordinateElementsByName["standard_name"] = "standard_name:string";

    asCoordinateElementsByField[0]       = "dummy";
    anCoordinateFieldsByElement["dummy"] = 0;

    asVariableElementsByName["@name"]                     = "@name:string";
    asVariableElementsByName["@shape"]                    = "@shape:string";
    asVariableElementsByName["@type"]                     = "@type:string";
    asVariableElementsByName["units"]                     = "units:string";
    asVariableElementsByName["valid_min"]                 = "valid_min:type";
    asVariableElementsByName["valid_max"]                 = "valid_max:type";
    asVariableElementsByName["_FillValue"]                = "_FillValue:type";
    asVariableElementsByName["missing_value"]             = "missing_value:type";
    asVariableElementsByName["long_name"]                 = "long_name:string";
    asVariableElementsByName["source"]                    = "source:string";
    asVariableElementsByName["coordinates"]               = "coordinates:string";
    asVariableElementsByName["standard_name"]             = "standard_name:string";
    asVariableElementsByName["flag_masks"]                = "flag_masks:type";
    asVariableElementsByName["flag_meanings"]             = "flag_meanings:string";
    asVariableElementsByName["flag_values"]               = "flag_values:type";
    asVariableElementsByName["ancillary_variables"]       = "ancillary_variables:string";
    asVariableElementsByName["cell_measures"]             = "cell_measures:string";
    asVariableElementsByName["cell_methods"]              = "cell_methods:string";
    asVariableElementsByName["comment"]                   = "comment:string";
    asVariableElementsByName["grid_mapping"]              = "grid_mapping:string";
    asVariableElementsByName["references"]                = "references:string";
    asVariableElementsByName["standard_error_multiplier"] = "standard_error_multiplier:type";

    # The AddOffset and ScaleFactor elements require a type different than the
    # type of the variable.  This must be specified with a ScaledType element.
    #
    asVariableElementsByName["add_offset"]   = "add_offset:scaled_type";
    asVariableElementsByName["scale_factor"] = "scale_factor:scaled_type";

    # These elements are for variables that contain coordinate data.
    #
    asVariableElementsByName["axis"]          = "axis:string";
    asVariableElementsByName["values"]        = "values:type";
    asVariableElementsByName["bounds"]        = "bounds:string";
    asVariableElementsByName["calendar"]      = "calendar:string";
    asVariableElementsByName["climatology"]   = "climatology:string";
    asVariableElementsByName["compress"]      = "compress:string";
    asVariableElementsByName["formula_terms"] = "formula_terms:string";
    asVariableElementsByName["leap_month"]    = "leap_month:int";
    asVariableElementsByName["leap_year"]     = "leap_year:int";
    asVariableElementsByName["month_lengths"] = "month_lengths:int";
    asVariableElementsByName["positive"]      = "positive:string";

    # The ScaledType element is special.  It specifies the type to use for the
    # AddOffset and ScaleFactor elements.  It is not a variable attribute.
    #
    asVariableElementsByName["scaled_type"] = "scaled_type:string";

    asVariableElementsByField[0]       = "dummy";
    anVariableFieldsByElement["dummy"] = 0;

    # Initialize variables used to handle insertion of global attributes used to
    # describe missing values used in the file.
    #
    bHaveMissingValues    = 0;
    bGlobalAttributesSeen = 0;
}


# Remove any quote characters or carriage-return characters from the input line.
#
{
    gsub("\"", "");
    gsub("\r", "");
}


# Skip any lines that begin with a "#" as the first non-whitespace character.
# This allows comment lines to be inserted.
#
$0 ~ /^[[:space:]]*#/ \
{
    next;
}


# Remove any lines that are composed entirely of whitespace.
#
$0 ~ /^[[:space:]]*$/ \
{
    next;
}


# If the first field is "SpreadsheetVersion", get the version number.
#
$1 == "SpreadsheetVersion" \
{
    sSpreadsheetVersion = $2;

    next;
}


# If the first field is "Settings", set the state to indicate it.
#
$1 == "Settings" \
{
    sInSection = "settings";

    next;
}


# Set up to process fill values from the "Settings" section.
#
$1 == "" && $2 == "FillValues" && sInSection ~ /^settings/ \
{
    # Get an array of the input line fields, starting with the third one.
    #
    buildInputArray(asNameByField, 3);

    # Convert the names to lower_bar format.
    #
    convertNamesToLowerBar(asNameByField);

    # Build field-to-name and name-to-field mapping arrays.
    #
    buildReverseMap(asNameByField, anFieldByName);

    # Check for valid header names.  If there is an error, quit.
    #
    nResult = checkHeaderNames(anFieldByName, asFillValueElementsByName);

    if (0 == nResult) exit 1;

    # Build the elements by field number array.
    #
    buildElementsByField(anFieldByName, asFillValueElementsByName, asFillValueElementsByField);

    # Build the field number by elements array.
    #
    buildReverseMap(asFillValueElementsByField, anFillValueFieldsByElement);

    # Delete the contents of the FillValuesByType array.
    #
    delete asFillValuesByType;

    # Set the section to "settings:fillvalues".
    #
    sInSection = "settings:fillvalues";

    next;
}


# Process a fill value settings entry line.
#
$1 == "" && $2 == "" && sInSection == "settings:fillvalues" \
{
    # Get an array of the input line fields.
    #
    buildInputArray(asInputArray);

    # If either the value or type field is missing, report it and quit.
    #
    nKey1 = anFillValueFieldsByElement["@type:string"];
    nKey2 = anFillValueFieldsByElement["@value:type"];

    sType  = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
    sValue = (nKey2 in asInputArray) ? asInputArray[nKey2] : "";

    if ("" == sType)
    {
        printf "Error: Unable to find the fill value type at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    if ("" == sValue)
    {
        printf "Error: Unable to find the fill value at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    # Verify that the type is valid.  It must be one of float, double, byte,
    # short, int, long, or the unsigned variants of the integer types.  If not,
    # warn the user and skip the entry.
    #
    if (sType !~ /^(float|double|((unsigned )?(byte|short|int|long)))$/)
    {
        printf "Warning: Fill values of type \"%s\" are not allowed.  Discarding line %d.\n", sType, NR > "/dev/stderr";

        next;
    }

    # Store the value into the FillValuesByType array indexed by the type.
    #
    asFillValuesByType[sType] = sValue;

    # Move on.
    #
    next;
}


# Set up to process missing values from the "Settings" section.
#
$1 == "" && $2 == "MissingValues" && sInSection ~ /^settings/ \
{
    # Get an array of the input line fields, starting with the third one.
    #
    buildInputArray(asNameByField, 3);

    asNameByField[3] = toLowerBar(asNameByField[3]);

    # Build field-to-name and name-to-field mapping arrays.
    #
    buildReverseMap(asNameByField, anFieldByName);

    # Check for valid header names.  If there is an error, quit.
    #
    nResult = checkHeaderNames(anFieldByName, asMissingValueElementsByName, "true", "true");

    if (0 == nResult) exit 1;

    # Build the elements by field number array.
    #
    buildElementsByField(anFieldByName, asMissingValueElementsByName, asMissingValueElementsByField);

    # Build the field number by elements array.
    #
    buildReverseMap(asMissingValueElementsByField, anMissingValueFieldsByElement);

    # Delete the contents of the MissingValueByType and MissingValueMeanings
    # arrays.
    #
    delete asMissingValueByType;
    delete asMissingValueMeanings;

    # Store the column headers after the Type column header, which are the
    # missing value meanings, into the MissingValueMeanings array in sequential
    # order from 1.
    #
    nKey1 = 1;

    for (nKey2 = 4 ; nKey2 <= NF ; nKey2++)
    {
        sValue = (nKey2 in asNameByField) ? asNameByField[nKey2] : "";

        if ("" != sValue)
        {
            asMissingValueMeanings[nKey1++] = sValue;
        }
    }

    # Set the section to "settings:missingvalues".
    #
    sInSection = "settings:missingvalues";

    next;
}


# Process a missing value settings entry line.
#
$1 == "" && $2 == "" && sInSection == "settings:missingvalues" \
{
    # Get an array of the input line fields.
    #
    buildInputArray(asInputArray);

    # Verify that there is an entry for each column header.  If not, report an
    # error and exit.
    #
    for (nKey1 in asMissingValueElementsByField)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        if ("" == sValue)
        {
            printf "Error: Column %d is empty in the missing values entry at line %d.\n", nKey1, NR > "/dev/stderr";

            exit 1;
        }
    }

    # Verify that there are no extraneous elements in the input array.  If there
    # are, warn the user and remove them.
    #
    for (nKey1 in asInputArray)
    {
        sValue = (nKey1 in asMissingValueElementsByField) ? asMissingValueElementsByField[nKey1] : "";

        if ("" == sValue)
        {
            printf "Warning: There is no MissingValues column header for field %d in line %d.  Discarding.\n", nKey1, NR > "/dev/stderr";

            asInputArray[nKey1] = "";
        }
    }

    # Get the type.
    #
    nKey1 = anMissingValueFieldsByElement["@type:string"];
    sType = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

    # Verify that the type is valid.  It must be one of float, double, byte,
    # short, int, long, or the unsigned variants of the integer types.  If not,
    # warn the user and skip the entry.
    #
    if (sType !~ /^(float|double|((unsigned )?(byte|short|int|long)))$/)
    {
        printf "Warning: missing values of type \"%s\" are not allowed.  Discarding line %d.\n", sType, NR > "/dev/stderr";

        next;
    }

    # Build a string composed of the missing value entries in order of
    # appearance, separated by spaces.  Skip empty columns.
    #
    sString = "";

    for (nKey1++ ; nKey1 <= NF ; nKey1++)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        if ("" != sValue)
        {
            # Don't put a space before the first value.
            #
            if ("" != sString)
            {
               sString = sString " ";
            }

            sString = sString sValue;
        }
    }

    # Store the value into the MissingValuesByType array indexed by the type.
    #
    asMissingValuesByType[sType] = sString;

    # Flag that missing values are available.
    #
    bHaveMissingValues = 1;

    # Move on.
    #
    next;
}


# If a new section is starting, the previous section was the global attributes
# section, and default missing values were set in the "Settings" section, append
# global attributes describing the missing values.
#
($1 != "" || $2 != "") && sInSection == "attributes" && bGlobalAttributesSeen && bHaveMissingValues \
{
    # Clear the HaveMissingValues and GlobalAttributesSeen flags.  This prevents
    # re-entry into this action.
    #
    bGlobalAttributesSeen = 0;
    bHaveMissingValues    = 0;

    # Create a string containing the missing value meanings separated by spaces.
    #
    nKey2 = length(asMissingValueMeanings);

    sValue = asMissingValueMeanings[1];

    for (nKey1 = 2 ; nKey1 <= nKey2 ; nKey1++)
    {
        sValue = sValue " ";
        sValue = sValue asMissingValueMeanings[nKey1];
    }

    # Add an attribute describing the missing value meanings.
    #
    addAttribute("missing_value_meanings:string", "string", sValue);

    # Write an attribute for each type that has a missing value string.
    #
    sValue = ("byte" in asMissingValuesByType) ? asMissingValuesByType["byte"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_8bit_signed:string", "string", sValue);
    }

    sValue = ("unsigned byte" in asMissingValuesByType) ? asMissingValuesByType["unsigned byte"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_8bit_unsigned:string", "string", sValue);
    }

    sValue = ("short" in asMissingValuesByType) ? asMissingValuesByType["short"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_16bit_signed:string", "string", sValue);
    }

    sValue = ("unsigned short" in asMissingValuesByType) ? asMissingValuesByType["unsigned short"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_16bit_unsigned:string", "string", sValue);
    }

    sValue = ("int" in asMissingValuesByType) ? asMissingValuesByType["int"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_32bit_signed:string", "string", sValue);
    }

    sValue = ("unsigned int" in asMissingValuesByType) ? asMissingValuesByType["unsigned int"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_32bit_unsigned:string", "string", sValue);
    }

    sValue = ("long" in asMissingValuesByType) ? asMissingValuesByType["long"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_64bit_signed:string", "string", sValue);
    }

    sValue = ("unsigned long" in asMissingValuesByType) ? asMissingValuesByType["unsigned long"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_64bit_unsigned:string", "string", sValue);
    }

    sValue = ("float" in asMissingValuesByType) ? asMissingValuesByType["float"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_32bit_float:string", "string", sValue);
    }

    sValue = ("double" in asMissingValuesByType) ? asMissingValuesByType["double"] : "";

    if ("" != sValue)
    {
        addAttribute("missing_values_64bit_float:string", "string", sValue);
    }
}


# A new group is starting.  Do post-processing for any previous group, and
# pre-processing for the new group.
#
$1 == "Group" \
{
    # If there is an existing parent group, save the current group contents into
    # the group contents array indexed by the current group path.
    #
    if ("" != sGroupPath)
    {
        asGroupContents[sGroupPath] = sGroupContents;
    }

    # Initialize for the new group.
    #
    nGroupDepth    = 0;
    sGroupContents = "";
    sGroupPath     = "Global";

    # If the new group is not the "Global" group, prepend "Global/" to the group
    # path.
    #
    if ("Global" != $2)
    {
        sGroupPath = sGroupPath "/" $2;
    }

    # Determine the depth of the new group path.
    #
    nElements   = split(sGroupPath, asElements, "/");
    nGroupDepth = nElements - 1;

    # If the group name is invalid, report the error and exit.
    #
    sValue = asElements[nElements];

    if (0 == verifyName(sValue))
    {
        printf "Error: Group name \"%s\" is invalid at line %d.\n", sValue, NR > "/dev/stderr";

        exit 1;
    }

    # Store away the depth, group name (last element of the path), and path into
    # arrays indexed by the group path.
    #
    anGroupDepth[sGroupPath] = nGroupDepth;
    asGroupName[sGroupPath]  = sValue;
    asGroupPath[sGroupPath]  = sGroupPath;

    # Set the section to "none" to be ready for the next section in the group.
    #
    sInSection = "none";

    # Set the indent information based on the group depth.
    #
    nIndentLevel = 4 * (nGroupDepth + 1);
    sIndent      = substr(sIndentSource, 1, nIndentLevel);

    next;
}


#Initialize for a new group attributes section.
#
$1 == "" && $2 == "Attributes" \
{
    # If these are the global attributes, flag that they have been seen.
    #
    if ("Global" == sGroupPath)
    {
        bGlobalAttributesSeen = 1;
    }

    # Get an array of the input line fields, starting with the third one.
    #
    buildInputArray(asNameByField, 3);

    # Convert the names to lower_bar format.
    #
    convertNamesToLowerBar(asNameByField);

    # Build field-to-name and name-to-field mapping arrays.
    #
    buildReverseMap(asNameByField, anFieldByName);

    # Check for valid header names.  If there is an error, quit.
    #
    nResult = checkHeaderNames(anFieldByName, asAttributeElementsByName);

    if (0 == nResult) exit 1;

    # Build the elements by field number array.
    #
    buildElementsByField(anFieldByName, asAttributeElementsByName, asAttributeElementsByField);

    # Build the field number by elements array.
    #
    buildReverseMap(asAttributeElementsByField, anAttributeFieldsByElement);

    # Set InSection state variable.
    #
    sInSection = "attributes";

    next;
}


#Initialize for a new group dimensions section.
#
$1 == "" && $2 == "Dimensions" \
{
    # Get an array of the input line fields, starting with the third one.
    #
    buildInputArray(asNameByField, 3);

    # Convert the names to lower_bar format.
    #
    convertNamesToLowerBar(asNameByField);

    # Build field-to-name and name-to-field mapping arrays.
    #
    buildReverseMap(asNameByField, anFieldByName);

    # Check for valid header names.  If there is an error, quit.
    #
    nResult = checkHeaderNames(anFieldByName, asDimensionElementsByName);

    if (0 == nResult) exit 1;

    # Build the elements by field number array.
    #
    buildElementsByField(anFieldByName, asDimensionElementsByName, asDimensionElementsByField);

    # Build the field number by elements array.
    #
    buildReverseMap(asDimensionElementsByField, anDimensionFieldsByElement);

    # Set the InSection state variable.
    #
    sInSection = "dimensions";

    next;
}


# Initialize for a new group coordinates section.
#
$1 == "" && $2 == "Coordinates" \
{
    # Get an array of the input line fields, starting with the third one.
    #
    buildInputArray(asNameByField, 3);

    # Convert the names to lower_bar format.
    #
    convertNamesToLowerBar(asNameByField);

    # Build field-to-name and name-to-field mapping arrays.
    #
    buildReverseMap(asNameByField, anFieldByName);

    # If there is only an "@name" field, add an "@shape" field to the end.  If
    # there is only an "@shape" field, add an "@name" field to the end.
    #
    if (("@name" in anFieldByName) && !("@shape" in anFieldByName))
    {
        anFieldByName["@shape"] = NF + 1;
    }
    else if (!("@name" in anFieldByName) && ("@shape" in anFieldByName))
    {
        anFieldByName["@name"] = NF + 1;
    }

    # Check for valid header names.  Allow missing and extra columns.  If there
    # is an error, quit.
    #
    nResult = checkHeaderNames(anFieldByName, asCoordinateElementsByName, "true", "true");

    if (0 == nResult) exit 1;

    # If an "@name" or "@shape" field was added, remove it.
    #
    if (NF + 1 == anFieldByName["@name"])
    {
        delete anFieldByName["@name"];
    }
    else if (NF + 1 == anFieldByName["@shape"])
    {
        delete anFieldByName["@shape"];
    }

    # Build the elements by field number array.
    #
    buildElementsByField(anFieldByName, asCoordinateElementsByName, asCoordinateElementsByField);

    # Build the field number by elements array.
    #
    buildReverseMap(asCoordinateElementsByField, anCoordinateFieldsByElement);

    # If there are fill or missing value column headers, warn the user that
    # these should not be applied to coordinates.  (By CF conventions,
    # coordinates are not allowed to have missing values.  This differs from
    # variables that are declared as auxiliary coordinates for other variables.)
    #
    if ("_FillValue:type" in anCoordinateFieldsByElement)
    {
        printf "Warning: CF conventions do not allow coordinates to have fill.  (At line %d.)\n", NR > "/dev/stderr";
    }

    if ("missing_value:type" in anCoordinateFieldsByElement)
    {
        printf "Warning: CF conventions do not allow coordinates to have missing values.  (At line %d.)\n", NR > "/dev/stderr";
    }

    # Coordinate variables have names that are the same as their shapes, which
    # must be one-dimensional.  If both name and shape columns are present, warn
    # the user.  If neither is present, report an error and exit.
    #
    nKey1 = ("@name:string" in anCoordinateFieldsByElement);
    nKey2 = ("@shape:string" in anCoordinateFieldsByElement);

    if (1 == nKey1 && 1 == nKey2)
    {
        printf "Warning: Coordinates column headers for name and shape both specified.  Entries in both fields must match.  (At line %d.)\n", \
               NR > "/dev/stderr";
    }
    else if (0 == nKey1 && 0 == nKey2)
    {
        printf "Error: Coordinates column headers must contain a name or a shape field.  (At line %d.)\n", NR > "/dev/stderr";

        exit 1;
    }

    # Set the InSection state variable.
    #
    sInSection = "coordinates";

    # Clear the list of coordinates names.
    #
    delete anCoordinateNames;

    next;
}


# Initialize for a new group variables section.
#
$1 == "" && $2 == "Variables" \
{
    # Get an array of the input line fields, starting with the third one.
    #
    buildInputArray(asNameByField, 3);

    # Convert the names to lower_bar format.
    #
    convertNamesToLowerBar(asNameByField);

    # Build field-to-name and name-to-field mapping arrays.
    #
    buildReverseMap(asNameByField, anFieldByName);

    # Check for valid header names.  Allow missing and extra columns.  If there
    # is an error, quit.
    #
    nResult = checkHeaderNames(anFieldByName, asVariableElementsByName, "true", "true");

    if (0 == nResult) exit 1;

    # Build the elements by field number array.
    #
    buildElementsByField(anFieldByName, asVariableElementsByName, asVariableElementsByField);

    # Build the field number by elements array.
    #
    buildReverseMap(asVariableElementsByField, anVariableFieldsByElement);

    # Set the InSection state variable.
    #
    sInSection = "variables";

    next;
}


# Process an attribute entry.
#
$1 == "" && $2 == "" && sInSection == "attributes" \
{
    # Get an array of the input line fields.
    #
    buildInputArray(asInputArray);

    # If the name, value, or type field is missing, report it and quit.
    #
    nKey1 = anAttributeFieldsByElement["@name:string"];
    nKey2 = anAttributeFieldsByElement["@type:string"];
    nKey3 = anAttributeFieldsByElement["@value:type"];

    sName = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

    if ("" == sName)
    {
        printf "Error: Unable to find the attribute name at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    sType = (nKey2 in asInputArray) ? asInputArray[nKey2] : "";

    if ("" == sType)
    {
        printf "Error: Unable to find the attribute type at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    sValue = (nKey3 in asInputArray) ? asInputArray[nKey3] : "";

    if ("" == sValue)
    {
        printf "Warning: the attribute \"%s\" has no value at line %d.\n", sName, NR > "/dev/stderr";
    }

    # If the name is invalid, report the error and exit.
    #
    if (0 == verifyName(sName))
    {
        printf "Error: Attribute name \"%s\" is invalid at line %d.\n", sName, NR > "/dev/stderr";

        exit 1;
    }

    # If the type is an unsigned type, warn the user that unsigned types for
    # attributes are not supported in NcML.  (This is because Java doesn't
    # understand unsigned types. Grrrrr!)
    #
    if (sType ~ /^unsigned /)
    {
        printf "Warning: Attribute %s has a type of \"%s\", which is unsupported for group attributes in NcML (Java limitation) at line %d\n", \
               sName, sType, NR > "/dev/stderr";
    }

    # If the type is "string", convert any characters that need converting into
    # XML escape sequences.
    #
    if ("string" == sType)
    {
        sValue = convertToXmlCharacters(sValue);
    }

    # Add the XML for the attribute to the group contents string.
    #
    sGroupContents = sprintf("%s%s<attribute name=\"%s\" type=\"%s\" value=\"%s\" />\n",
                             sGroupContents, sIndent, sName, sType, sValue);

    next;
}


# Process a dimension entry.
#
$1 == "" && $2 == "" && sInSection == "dimensions" \
{
    # Get an array of the input line fields.
    #
    buildInputArray(asInputArray);

    # If the name or length field is missing, report it and quit.
    #
    nKey1 = anDimensionFieldsByElement["@name:string"];
    nKey2 = anDimensionFieldsByElement["@length:string"];

    sName = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

    if ("" == sName)
    {
        printf "Error: Unable to find the dimension name at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    sLength = (nKey2 in asInputArray) ? asInputArray[nKey2] : "";

    if ("" == sLength)
    {
        printf "Error: Unable to find the dimension length at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    # If the name is invalid, report the error and exit.
    #
    if (0 == verifyName(sName))
    {
        printf "Error: Dimension name \"%s\" is invalid at line %d.\n", sName, NR > "/dev/stderr";

        exit 1;
    }

    # If the name has already been defined at this level, report the error and
    # exit.
    #
    if ((sGroupPath "/" sName) in anDimensionDictionary)
    {
        printf "Error: Dimension name \"%s\" being redefined in group \"%s\" at line %d.\n", sName, sGroupPath, NR > "/dev/stderr";

        exit 1.
    }

    # If the value of the length is "unlimited" or "0", set the length to 0 and
    # set the IsUnlimited flag to true.  Otherwise set the length to the numeric
    # value of the length string and set the IsUnlimited flag to false.
    #
    if ("unlimited" == sLength || "0" == sLength)
    {
        nLength      = 0;
        sIsUnlimited = "true";
    }
    else
    {
        nLength      = sLength + 0;
        sIsUnlimited = "false";
    }

    # Add the XML for the dimension to the group contents string.
    #
    sGroupContents = sprintf("%s%s<dimension name=\"%s\" length=\"%d\" isUnlimited=\"%s\" />\n",
                             sGroupContents, sIndent, sName, nLength, sIsUnlimited);

    # Add the dimension to the DimensionDictionary by forming a path using the
    # current group path, a "/", and the dimension name as the key, and setting
    # the value to 1.
    #
    anDimensionDictionary[sGroupPath "/" sName] = 1;

    next;
}


# Process a coordinate entry.
#
$1 == "" && $2 == "" && sInSection == "coordinates" \
{
    # Get an array of the input line fields.
    #
    buildInputArray(asInputArray);

    # Get the name, shape, and type field contents.
    #
    nKey1 = anCoordinateFieldsByElement["@name:string"];
    nKey2 = anCoordinateFieldsByElement["@shape:string"];
    nKey3 = anCoordinateFieldsByElement["@type:string"];

    sName     = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
    sShape    = (nKey2 in asInputArray) ? asInputArray[nKey2] : "";
    sFullType = (nKey3 in asInputArray) ? asInputArray[nKey3] : "";

    # If neither a name or a shape was specified, report the error and exit.
    # If both were specified and they don't match, report the error and exit.
    # 
    if ("" == sName && "" == sShape)
    {
        printf "Error: Unable to find the coordinate name or shape at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }
    else if ("" != sName && "" != sShape && sName != sShape)
    {
        printf "Error: The coordinate name and shape conflict at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    # If the shape wasn't specified, set it equal to the name.
    #
    if ("" == sShape)
    {
        sShape = sName;
    }

    # If the name wasn't specified, set it equal to the shape.
    #
    if ("" == sName)
    {
        sName = sShape;
    }

    # If the type wasn't specified, report an error and exit.
    #
    if ("" == sFullType)
    {
        printf "Error: Unable to find the coordinate type at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    # If the name is invalid, report the error and exit.
    #
    if (0 == verifyName(sName))
    {
        printf "Error: Coordinate name \"%s\" is invalid at line %d.\n", sName, NR > "/dev/stderr";

        exit 1;
    }

    # If the shape isn't valid, exit.
    #
    if (0 == isShapeValid(sShape))
    {
        exit 1;
    }

    # If the type string is prefixed by "unsigned ", remove the prefix and set
    # the IsUnsigned flag.  Otherwise, clear the IsUnsigned flag.
    #
    bIsUnsigned = 0;
    sType       = sFullType;

    if (sFullType ~ /^unsigned /)
    {
        sType       = substr(sFullType, 10);
        bIsUnsigned = 1;
    }

    # Add the XML variable entry start for the coordinate to the group contents string.
    #
    sGroupContents = sprintf("%s%s<variable name=\"%s\" shape=\"%s\" type=\"%s\">\n",
                             sGroupContents, sIndent, sName, sShape, sType);

    # Increase the indent level to prepare for adding any variable attributes.
    #
    nIndentLevel += 4;
    sIndent       = substr(sIndentSource, 1, nIndentLevel);

    # If the type is unsigned, add the XML for the _Unsigned attribute.
    #
    if (0 != bIsUnsigned)
    {
        sGroupContents = sprintf("%s%s<attribute name=\"_Unsigned\" type=\"string\" value=\"true\" />\n",
                                 sGroupContents, sIndent);
    }

    # Remove the name, shape, and type elements from the input.
    #
    delete asInputArray[nKey1];
    delete asInputArray[nKey2];
    delete asInputArray[nKey3];

    # If there is a "values" entry, build the XML for the values element and
    # remove it from the input.
    #
    nKey1  = ("values:type" in anCoordinateFieldsByElement) ? anCoordinateFieldsByElement["values:type"] : "";

    sValuesElement = "";

    if ("" != nKey1)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        if ("" != sValue)
        {
            # If the type is "string", convert any characters that need converting into
            # XML escape sequences.
            #
            if ("string" == sType)
            {
                sValue = convertToXmlCharacters(sValue);
            }

            # Build the XML values element string from the input.  (This will
            # write a warning message if there was a problem.)
            #
            sValuesElement = buildValuesElement(sValue);

            delete asInputArray[nKey1];
        }
    }

    # Write attribute entries for all remaining elements of InputArray.  If an
    # element doesn't have a header entry, write a warning and skip it.
    #
    for (nKey1 in asInputArray)
    {
        # Get the coordinate element info for the field.  If there is none, warn
        # the user and skip to the next element of InputArray.
        #
        sElement = (nKey1 in asCoordinateElementsByField) ? asCoordinateElementsByField[nKey1] : "";

        if ("" == sElement)
        {
            printf "Warning: There is no Coordinates column header for field %d in line %d.  Discarding.\n", nKey1, NR > "/dev/stderr";
            continue;
        }

        # Create the attribute entry.
        #
        addAttribute(sElement, sType, asInputArray[nKey1]);
    }

    # If there is a non-empty values element string, add it to the output.
    #
    if ("" != sValuesElement)
    {
        sGroupContents = sprintf("%s%s%s\n", sGroupContents, sIndent, sValuesElement);
    }

    # Reduce the indent level.
    #
    nIndentLevel -= 4;
    sIndent       = substr(sIndentSource, 1, nIndentLevel);

    # End the XML variable entry.
    #
    sGroupContents = sprintf("%s%s</variable>\n",
                             sGroupContents, sIndent);

    # Add the name of the coordinate variable to the list of coordinate names.
    #
    anCoordinateNames[sName] = 1;

    next;
}


# Process a regular variable entry.
#
$1 == "" && $2 == "" && sInSection == "variables" \
{
    # Get an array of the input line fields.
    #
    buildInputArray(asInputArray);

    # If the name or type field is missing, report it and quit.  Get the shape
    # field, but it is OK for it to be empty.  An empty or zero-valued shape
    # is an indicator of a scalar variable.
    #
    nKey1 = anVariableFieldsByElement["@name:string"];
    nKey2 = anVariableFieldsByElement["@shape:string"];
    nKey3 = anVariableFieldsByElement["@type:string"];

    sName = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

    if ("" == sName)
    {
        printf "Error: Unable to find the variable name at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    sShape = (nKey2 in asInputArray) ? asInputArray[nKey2] : "";

    sFullType = (nKey3 in asInputArray) ? asInputArray[nKey3] : "";

    if ("" == sFullType)
    {
        printf "Error: Unable to find the variable type at line %d.\n", NR > "/dev/stderr";

        exit 1;
    }

    # If the name is invalid, report the error and exit.
    #
    if (0 == verifyName(sName))
    {
        printf "Error: Variable name \"%s\" is invalid at line %d.\n", sName, NR > "/dev/stderr";

        exit 1;
    }

    # If the shape isn't empty or zero, verify that it is valid.  If not, exit.
    #
    if (sShape)
    {
        if (0 == isShapeValid(sShape))
        {
            exit 1;
        }
    }

    # If the type string is prefixed by "unsigned ", remove the prefix and set
    # the IsUnsigned flag.  Otherwise, clear the IsUnsigned flag.
    #
    bIsUnsigned = 0;
    sType       = sFullType;

    if (sFullType ~ /^unsigned /)
    {
        sType       = substr(sFullType, 10);
        bIsUnsigned = 1;
    }

    # Add the XML variable entry start to the group contents string.
    #
    sGroupContents = sprintf("%s%s<variable name=\"%s\" shape=\"%s\" type=\"%s\">\n",
                             sGroupContents, sIndent, sName, sShape, sType);

    # Increase the indent level to prepare for adding any variable attributes.
    #
    nIndentLevel += 4;
    sIndent       = substr(sIndentSource, 1, nIndentLevel);

    # If the type is unsigned, add the XML for the _Unsigned attribute.
    #
    if (0 != bIsUnsigned)
    {
        sGroupContents = sprintf("%s%s<attribute name=\"_Unsigned\" type=\"string\" value=\"true\" />\n",
                                 sGroupContents, sIndent);
    }

    # Remove the name, shape, and type elements from the input.
    #
    delete asInputArray[nKey1];
    delete asInputArray[nKey2];
    delete asInputArray[nKey3];

    # If there is a "values" entry, add the XML for the values element, then
    # remove it from the input.
    #
    nKey1  = ("values:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["values:type"] : "";

    sValuesElement = "";

    if ("" != nKey1)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        if ("" != sValue)
        {
            # If the type is "string", convert any characters that need converting into
            # XML escape sequences.
            #
            if ("string" == sType)
            {
                sValue = convertToXmlCharacters(sValue);
            }

            # Build the XML values element string from the input.  (This will
            # write a warning message if there was a problem.)
            #
            sValuesElement = buildValuesElement(sValue);

            delete asInputArray[nKey1];
        }
    }

    # If the variable type is "string", and any fill, missing, valid min/max,
    # or scaled output attributes were specified, warn the user and remove
    # them.  Othewise, deal with supplying defaults for the fill, missing, and
    # min/max attributes.
    #
    if ("string" == sType)
    {
        nKey1 = ("_FillValue:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["_FillValue:type"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: _FillValue attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }
        
        nKey1 = ("valid_min:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["valid_min:type"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: valid_min attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }
        
        nKey1 = ("valid_max:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["valid_max:type"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: valid_max attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }
        
        nKey1 = ("missing_value:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["missing_value:type"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: missing_value attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }

        nKey1 = ("scaled_type:string" in anVariableFieldsByElement) ? anVariableFieldsByElement["scaled_type:string"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: scaled_type attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }

        nKey1 = ("add_offset:scaled_type" in anVariableFieldsByElement) ? anVariableFieldsByElement["add_offset:scaled_type"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: add_offset attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }

        nKey1 = ("scale_factor:scaled_type" in anVariableFieldsByElement) ? anVariableFieldsByElement["scale_factor:scaled_type"] : "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

            if ("" != sValue)
            {
                printf "Warning: scale_factor attribute specified for a variable of type string in line %d.  Discarding.\n", NR > "/dev/stderr";

                delete asInputArray[nKey1];
            }
        }
    }
    else
    {
        # If there is no fill value entry, add the appropriate default for the
        # type if it exists.  Otherwise, add the specified fill value entry and
        # remove it from the input array.  If the fill value entry is "n/a",
        # don't add a fill value attribute.
        #
        nKey1  = ("_FillValue:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["_FillValue:type"] : "";
        sValue = "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
        }

        if ("" == sValue)
        {
            sValue = (sFullType in asFillValuesByType) ? asFillValuesByType[sFullType] : "";
        }
        else if ("" != nKey1)
        {
            delete asInputArray[nKey1];
        }

        if ("" != sValue && "n/a" != sValue)
        {
            addAttribute("_FillValue:type", sType, sValue);
        }

        # If there is no valid_min entry, add the appropriate default for the
        # type, if it exists.  Otherwise, add the specified valid_min entry and
        # remove it from the input array.
        #
        nKey1  = ("valid_min:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["valid_min:type"] : "";
        sValue = "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
        }

        if ("" == sValue)
        {
            sValue = (sFullType in asValidMinByType) ? asValidMinByType[sFullType]: "";
        }
        else if ("" != nKey1)
        {
            delete asInputArray[nKey1];
        }

        if ("" != sValue)
        {
            addAttribute("valid_min:type", sType, sValue);
        }

        # If there is no valid_max entry, add the appropriate default for the
        # type if it exists.  Otherwise, add the specified valid_max entry and
        # remove it from the input array.
        #
        nKey1  = ("valid_max:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["valid_max:type"] : "";
        sValue = "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
        }

        if ("" == sValue)
        {
            sValue = (sFullType in asValidMaxByType) ? asValidMaxByType[sFullType] : "";
        }
        else if ("" != nKey1)
        {
            delete asInputArray[nKey1];
        }

        if ("" != sValue)
        {
            addAttribute("valid_max:type", sType, sValue);
        }

        # If there is no missing_value entry, and there is a default value
        # specified, add the appropriate default for the type.  Otherwise, add
        # the specified missing_value entry and remove it from the input array.
        # If the missing value entry is "n/a", don't add the attribute.
        #
        nKey1  = ("missing_value:type" in anVariableFieldsByElement) ? anVariableFieldsByElement["missing_value:type"] : "";
        sValue = "";

        if ("" != nKey1)
        {
            sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
        }

        if ("" == sValue)
        {
            sValue = asMissingValuesByType[sFullType];
        }
        else if ("" != nKey1)
        {
            delete asInputArray[nKey1];
        }

        if ("" != sValue && "n/a" != sValue)
        {
            addAttribute("missing_value:type", sType, sValue);
        }
    }

    # If there is a "scaled_type" entry, get the value and remove the element
    # from the input array.
    #
    nKey1       = ("scaled_type:string" in anVariableFieldsByElement) ? anVariableFieldsByElement["scaled_type:string"] : "";
    sScaledType = "";

    if ("" != nKey1)
    {
        sScaledType = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        delete asInputArray[nKey1];
    }

    # If there is an "add_offset" entry, create an attribute for it and remove
    # the element from the input array.  If no scaled type was found, report an
    # error and exit.
    #
    nKey1  = ("add_offset:scaled_type" in anVariableFieldsByElement) ? anVariableFieldsByElement["add_offset:scaled_type"] : "";

    if ("" != nKey1)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        # If there is a value, remove the element from the input array and
        # attempt to create an attribute.
        #
        if ("" != sValue)
        {
            delete asInputArray[nKey1];

            # If there was no scaled type, report the error and exit.
            #
            if ("" == sScaledType)
            {
                printf "Error: No scaled type specified for the add_offset attribute (field %d) in line %d.\n", nKey1, NR > "/dev/stderr";

                exit 1;
            }

            # Create the attribute.
            #
            addAttribute("add_offset:type", sScaledType, sValue);
        }
    }

    # If there is a "scale_factor" entry, create an attribute for it and remove
    # the element from the input array.  If no scaled type was found, report an
    # error and exit.
    #
    nKey1  = ("scale_factor:scaled_type" in anVariableFieldsByElement) ? anVariableFieldsByElement["scale_factor:scaled_type"] : "";

    if ("" != nKey1)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";

        # If there is a value, remove the element from the input array and
        # attempt to create an attribute.
        #
        if ("" != sValue)
        {
            delete asInputArray[nKey1];

            # If there was no scaled type, report the error and exit.
            #
            if ("" == sScaledType)
            {
                printf "Error: No scaled type specified for the scale_factor attribute (field %d) in line %d.\n", nKey1, NR > "/dev/stderr";

                exit 1;
            }

            # Create the attribute.
            #
            addAttribute("scale_factor:type", sScaledType, sValue);
        }
    }

    # Get any coordinates attribute that was specified and remove it from the
    # input array.
    #
    nKey1  = ("coordinates:string" in anVariableFieldsByElement) ? anVariableFieldsByElement["coordinates:string"] : "";
    sValue = "";

    if ("" != nKey1)
    {
        sValue = (nKey1 in asInputArray) ? asInputArray[nKey1] : "";
    }

    if (sValue)
    {
        delete asInputArray[nKey1];
    }

    # Build a coordinates attribute value from the shape and any
    # coordinates attribute value that was specified.
    #
    sValue = buildCoordinatesAttribute(anCoordinateNames, sShape, sValue);

    # If the coordinate attribute value is not empty, create the attribute.
    #
    if ("" != sValue)
    {
        # Create the attribute.
        #
        addAttribute("coordinates:string", "string", sValue);
    }

    # Write attribute entries for all remaining elements of InputArray.  If an
    # element doesn't have a header entry, write a warning and skip it.
    #
    for (nKey1 in asInputArray)
    {
        # Get the value.  If it is empty, skip to the next element.
        #
        sValue = asInputArray[nKey1];

        if ("" == sValue)
        {
            continue;
        }

        # Get the variable element info for the field.  If there is none, warn
        # the user and skip to the next element of InputArray.
        #
        sElement = (nKey1 in asVariableElementsByField) ? asVariableElementsByField[nKey1] : "";

        if ("" == sElement)
        {
            printf "Warning: There is no Variables column header for field %d in line %d.  Discarding.\n", nKey1, NR > "/dev/stderr";
            continue;
        }

        # Create the attribute entry.
        #
        addAttribute(sElement, sType, sValue);
    }

    # If there is a non-empty values element string, add it to the output.
    #
    if ("" != sValuesElement)
    {
        sGroupContents = sprintf("%s%s%s\n", sGroupContents, sIndent, sValuesElement);
    }

    # Reduce the indent level.
    #
    nIndentLevel -= 4;
    sIndent       = substr(sIndentSource, 1, nIndentLevel);

    # End the XML variable entry.
    #
    sGroupContents = sprintf("%s%s</variable>\n", sGroupContents, sIndent);

    next;
}


# Write out the NcML document file using the information gathered into the group
# contents array.
#
END \
{
    # If there is a current group path set (and there should be), add the
    # current group contents to the group contents array, indexed by the group
    # path.
    #
    if ("" != sGroupPath)
    {
        asGroupContents[sGroupPath] = sGroupContents;
    }

    # Sort the group path array.  The elements of the array will be ordered
    # alphanumerically.  Sub-groups will follow directly after parent groups.
    # This also gives us a count of the different groups.
    #
    nGroups = asort(asGroupPath);

    # Write out the head section of the document.
    #
    printf "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    printf "<netcdf xmlns=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2\" ";
    printf "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" ";
    printf "xsi:schemaLocation=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2 ";
    printf "http://www.unidata.ucar.edu/schemas/netcdf/ncml-2.2.xsd\">\n";

    # Start at a group depth of 0.  Set the initial indent level.
    #
    nGroupDepth = 0;

    nIndentLevel = 4;
    sIndent      = substr(sIndentSource, 1, nIndentLevel);

    # Process each group.
    #
    for (nGroup = 1 ; nGroup <= nGroups ; nGroup++)
    {
        # Get the path for the group.
        #
        sGroupPath = asGroupPath[nGroup];

        # If this is a sub-group, handle any unclosed sub-groups preceding this
        # one.
        #
        nDepth = anGroupDepth[sGroupPath];

        if (0 != nDepth)
        {
            # Decrement the indent level and write an XML group end element
            # until the group depth matches the current group depth. 
            #
            for (; nDepth <= nGroupDepth ; nGroupDepth--)
            {
                nIndentLevel -= 4;
                sIndent       = substr(sIndentSource, 1, nIndentLevel);

                printf "%s</group>\n", sIndent;
            }

            # Write a new group element and increase the indent level.
            #
            printf "%s<group name=\"%s\">\n", sIndent, asGroupName[sGroupPath];

            nIndentLevel += 4;
            sIndent       = substr(sIndentSource, 1, nIndentLevel);
        }

        # Set the new group depth value and write the group contents.
        #
        nGroupDepth = nDepth;

        printf "%s", asGroupContents[sGroupPath];
    }

    # Write XML group end elements to close all open groups, decrementing the
    # indent level each time.
    #
    for (nDepth = 0 ; nDepth < nGroupDepth ; nDepth++)
    {
        nIndentLevel -= 4;
        sIndent       = substr(sIndentSource, 1, nIndentLevel);

        printf "%s</group>\n", sIndent;
    }

    # Write the XML netcdf end element.
    #
    printf "</netcdf>\n";
}
